/*** HELP START ***//*
 
## >>> `%getTitle()` macro: <<< <a name="gettitle-macro"></a> #######################  

The getTitle() macro extract text of titles or footnotes
into a delimited list.

Titles/footnotes numbers can be selected with the `number` argument.
Only the text of a title or footnote is extracted.

Author of the original code is: Quentin McMullen (`qmcmullen.sas@gmail.com`).

See examples below for the details.

The `%getTitle()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%getTitle(
  < number>
  <,type=> 
  <,dlm=>
  <,qt=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `number`       - *Optional*, default value is empty,
                    indicates numbers of titles to be extracted.
                    Space-separated list is expected.
                    If empty or `_ALL_` extract all non-missing.

*. `type`         - *Optional*, default value is `T`.
                    Indicates which type is the searched.
                    `T` fro title, `F` for footnote.

*. `dlm`          - *Optional*, "DeLiMiter", default value is `|` (pipe).
                    Possible values are: `| \ / , . ~ * # @ ! + - _ : ?`
                    or `s` for space, `c` for comma, `d` for semicolon.

*. `qt`           - *Optional*, "QuoTes", default value is empty.
                    Use `%str()` for single quote symbol (e.g. `%str(%")`).
                    If there are multiple symbols, only the first and the 
                    second are selected as a leading and trailing one, 
                    e.g. `qt=""` gives `"title1 text" "title2 text" ... `.

---

*//*** HELP END ***/

/*
by Quentin McMullen <qmcmullen.sas@gmail.com>
*/

/* gettitle.sas */

%macro getTitle
   (number        /*Number 1-10, _ALL_(or empty), or space delimited list of numbers*/
   ,type=T        /*T for titles, F for footnote*/
   ,dlm=|         /*delimiter for list of titles*/
   ,qt=
   )/minoperator;

   %local where dsid rc ldlm titletext tt q1 q2;

   /* Input verification: */

   /* 1) set type */
   %if not(%superq(Type) in (T t F f)) %then
     %do;
       %let Type=T;
     %end;
   %let where=Type="%upcase(&Type)" ;


   /* 2) set delimiter */
   %if %superq(dlm)= %then
     %do;
       %let dlm=;
     %end;
   %else
     %do;
       %if not(%superq(dlm) in %str(S s C c D d | \ / , . ~ * # @ ! + - _ : ?)) %then
         %do;
           %put NOTE: Unknown delimiter value %superq(dlm). Setting to default "|" (pipe)!;
           %put NOTE- Possible values are: %str(| \ / , . ~ * # @ ! + - _ : ?);
           %put NOTE- or s for space, c for comma, d for semicolon;
           %let dlm=|;
         %end;
       %if %superq(dlm) in (C c) %then
         %do;
           %let dlm=,;
         %end;
       %if %superq(dlm) in (D d) %then
         %do;
           %let dlm=%str(;);
         %end;
       %if %superq(dlm) in (S s) %then
         %do;
           %let dlm=%str( );
         %end;
     %end;

   /* 3) set quotes */
   %if %superq(qt)= %then
     %do;
       %let q1=;
       %let q2=;
     %end;
  %else
    %do;
      %if %length(&qt.)=1 %then
        %do;
          %let q1=&qt.;
          %let q2=&qt.;
        %end;
      %else
        %do;
          %let q1=%qsubstr(&qt.,1,1);
          %let q2=%qsubstr(&qt.,2,1);
        %end;
    %end;

   /* 4) select number of title */
   %if %superq(Number)= %then 
     %do ;
       %let Number=_ALL_;
     %end ;
   %if %qupcase(&Number) ne _ALL_ %then 
     %do ;
       %let where=&where and Number IN (&Number) ;
     %end ;

   /* extract data */
   %let dsid = %sysfunc ( open ( sashelp.vtitle(where=(&where)) ) ) ;

   %if &dsid > 0 %then %do ;
      %do %until ( &rc ^= 0 ) ;
         %let rc = %sysfunc ( fetch ( &dsid ) ) ;
         %if &rc=0 %then %do ;
            %let tt = %sysfunc ( getvarc ( &dsid , 3 ) );
            %if %superq(tt) = %then %let tt=%str( );
            %let TitleText =&TitleText.&ldlm.&q1.%superq(tt)&q2. ;
            %let ldlm=&dlm ;
         %end ;
      %end ;
      %let dsid = %sysfunc(close ( &dsid ) ) ;
   %end ;
   %else %put ERROR: (%nrstr(%%)&sysmacroname) could not open sashelp.vtitle ;

&TitleText. /*return*/
%mend getTitle ;



/* test cases:
title1 j=c "Hi Roger" ;
title2 j=l "Good Morning" ;
title3 "How are you?" ;
title4 ;
title5 "Bye bye!" ;

%put %GetTitle() ;

%put %GetTitle(,dlm=W) ;

%put %GetTitle(,dlm=C) ;
%put %GetTitle(_ALL_,dlm=%str( )) ;

%put %GetTitle(,dlm=s, qt=*) ;

%put %GetTitle(,dlm=s, qt="") ;
%put %GetTitle(,dlm=c, qt='') ;
%put %GetTitle(,dlm=d, qt=[]) ;

%put %GetTitle(,dlm=, qt=<>) ;

%put %GetTitle(1 3,dlm=c, qt='') ;

%put %GetTitle(2:4,dlm=c, qt='') ;

%put %GetTitle(4 5,dlm=c, qt='') ;

%put %GetTitle(17 42 303,dlm=c, qt='') ;

title;

%put %GetTitle() ;
data _null_;run;
*/

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Get titles in different forms:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  title1 j=c "Hi Roger" ;
  title2 j=l "Good Morning" ;
  title3 "How are you?" ;
  title4 ;
  title5 "Bye bye!" ;

  %put %GetTitle() ;

  %put %GetTitle(1 3,dlm=c, qt=[]) ;

  %put %GetTitle(2:4,dlm=s, qt='') ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.**  Get footnotes in different forms:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  footnote1 "First little footnote";
  footnote2 "Second little footnote";
  footnote3 "Third little footnote";

  %put %GetTitle(1 2,type=f,dlm=s, qt="") ;
  %put %GetTitle(2 3,type=f,dlm=c, qt='') ;
  %put %GetTitle(1 3,type=f,dlm=d, qt=[]) ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Quentin McMullen, Bartosz Jablonski, since 2023.         */
/*                                                                     */
/*  Code is under the MIT license. If you want - you can use it.       */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention authors:  */
/*  Quentin McMullen (qmcmullen.sas@gmail.com)                         */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
