/*** HELP START ***//*
 
## >>> `%iffunc()` macro: <<< <a name="iffunc-macro"></a> #######################  

The iffunc() macro function is a "lazy typer" wrapping up conditional execution.

Instead typing:
~~~~~~~~~~~~~~~~~~~~~~~sas
%let x = 1;

%if (1=&x.) %then
  %do;
    %let test = TRUE;
  %end;
%else 
  %do;
    %let test = FALSE;
  %end;

%put &=test.;
~~~~~~~~~~~~~~~~~~~~~~~

you can make it:

~~~~~~~~~~~~~~~~~~~~~~~sas
%let x = 1;

%let test = %iffunc((1=&x.),TRUE,FALSE);

%put &=test.;
~~~~~~~~~~~~~~~~~~~~~~~

The `%iffunc()` macro ca be nested in open code 
in contrary to standard `%IF-%THEN-%ELSE`.

The `minoperator` option is used to handle `IN` operator,
but the `IN` operator in the condition *has* to be surrounded 
by spaces, e.g. `&x. in (1 2 3)`. 
Condition of the form `&x. in(1 2 3)` (no space after `in`) will
result with an error.

See examples below for the details.

The `%iffunc()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%iffunc(
   cond
  ,true
  ,false
 <,v=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `cond`  - *Required*, is a condition to be evaluated.
             It **has** to be provided in brackets,
             e.g., `(1=&x.)` or `(%scan(&x.,1)=A)`.

2. `true`  - *Required*, value returned when condition is true.

3. `false` - *Required*, value returned when condition is false.

* `v=`     - *Optional*, v for verbose. If set to `1` extra notes
              are printed.
---

*//*** HELP END ***/

%macro iffunc(
  cond /* condition to be checked, has to be provided in brackets, e.g. `(&x.=42)` */
, true /* value returned when condition is true */
, false /* value returned when condition is false */
, v=0    /* v for verbose */
) 
/ 
parmbuff
minoperator
;
%let v = %sysevalf(1=%superq(v));
%if &v. %then 
  %do;
    %put [ifFunc] * START ****************************;
    %put %superq(syspbuff);
  %end;
/* the %sysevalf() does not handle IN operator so 
   it has to be handled separately 
 */
%if NOT (%str(IN) IN %qupcase(%superq(cond))) %then
  %do;
    %let cond = (%sysevalf(&cond));
  %end;
%if (&cond.) %then
  %do;&true.%end;
%else
  %do;&false.%end;
%if &v. %then %put [ifFunc] * END ******************************;  
%mend iffunc;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Simple test:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let x = 1;
  %let test = %iffunc((&x.=1),True,False);
  %put &=test.;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Verbose with `v=` in the `%put` statement:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put %iffunc((TRUE=true),TRUE,FALSE,v=1);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Condition can be with decimals or text:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put True: %iffunc((9.9<10),TRUE,FALSE);

  %put False: %iffunc((A>B),TRUE,FALSE);

  %put True: %iffunc((1=1.0),TRUE,FALSE);

  %put False: %iffunc((1/3 = 0.5),TRUE,FALSE);

  %put True: %iffunc((.1+.1+.1+.1+.1+.1+.1+.1+.1+.1 = 1),TRUE,FALSE);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Macro-Functions works too:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let x = A B C;
  %put %iffunc((%scan(&x.,1)=A),Starts with "A"., Does not start with "A".);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** Simple nesting (in open code):
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let x=0;
  %let y=0;

  %put
    %iffunc((&x.)
      ,It is X.
      ,%iffunc((&y.)
        ,It is Y.
        ,It is Z.
      ,v=1)
    ,v=1)
  ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6.** Special characters have to be masked:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put %iffunc(1,%str(,),%str(;));
  %put %iffunc(0,%str(,),%str(;));
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 7.** If you want to call macros there is a price. 
  The `%nrstr()` and `%unquote()` are required:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %macro doX(a,b);
    %put >>> &a. &b;
    data do_X;
      set sashelp.class;
    run;
  %mend;
  %macro doY(a,b);
    %put ### &a. &b;
    data do_Y;
      set sashelp.cars;
    run;
  %mend;
  %macro doZ(a,b);
    %put @@@ &a. &b;
    data do_Z;
      set sashelp.iris;
    run;
  %mend;

  %let x=0;
  %let y=0;

  %unquote(
  %iffunc((&x.)
    ,%nrstr(%doX(1,2))
    ,%iffunc((&y.)
      ,%nrstr(%doY(3,4))
      ,%nrstr(%doZ(5,6))
    ,v=1)
  ,v=1)
  )
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 8.** Traffic lights with `NOTE:` and `ERROR:`,
  Remember to handle the `IN` operator with an extra care.
  Spaces are required around the `IN` operator.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let x = A;
  %put %iffunc((&x. IN (A B C)),NOTE:,ERROR:) x=%superq(x);

  %let x = 7;
  %put %iffunc((&x. IN (1 3 5)),NOTE:,ERROR:) x=%superq(x);

  %let x = 1.1;
  %put %iffunc((&x. IN (1.1 2.1 3.1)),NOTE:,ERROR:) x=%superq(x);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 9.** Mixing `IN` operator with non-integer evaluation
  requires `%sysevalf()` function:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let x = Y;
  %let y = 9.9;

  %put THIS will be False: %iffunc((&x. in (y Y) AND &y.<10),TRUE,FALSE);

  %put THIS will be True: %iffunc((&x. in (y Y) AND %sysevalf(&y.<10)),TRUE,FALSE);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since 2024.                           */
/*                                                                     */
/*  Code is under the MIT license. If you want - you can use it.       */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/


/*
%put %iffunc(( ),test is TRUE,test is FALSE);

%put %iffunc(,test is TRUE,test is FALSE);
*/

