
/*monthshift.sas*/

/*** HELP START ***//*
 
## >>> `%monthShift()` macro: <<< <a name="monthshift-macro"></a> #######################  

The monthShift() macro is a utility macro 
which allows to shift "year-month" period by 
a given number of "periods" (months).

The result is in the `YYYYMM` format but can be altered.

See examples below for the details.

The `%monthShift()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%monthShift(
  < Y>
  <,M>
  <,shift>
  <,ofmt=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `Y`           - *Optional*, a year from which counting starts.
                   If null the value is set to *system year*.

2. `M`           - *Optional*, a month from which counting starts.
                   If null the value is set to `1`. Can be a number 
                   (`1` to `12`) or a name (`June`, `OCTOBER`) or 
                   a three letters short (`JAN`, `apr`).

3. `shift`       - *Optional*, number of periods to shift.
                   If null the value is set to `0`.
                   Positive value shifts to the "future",
                   negative value shifts to the "past",
                   Can be an expression (e.g. `1+2*3`, see examples).

* `ofmt=YYMMn6.` - *Optional*, it is a format name used to 
                   display the result. Default value is `YYMMn6.`
                   See examples.

---

*//*** HELP END ***/

%macro monthShift(Y,M,shift,ofmt=YYMMn6.);
%if %superq(Y) = %then %let Y = %sysfunc(year(%sysfunc(date())));
%if %superq(M) = %then %let M = 1;
%let Y = %sysfunc(inputn(%substr(%str(&M.   ), 1, 3)-&Y., ANYDTDTE.));
%if %superq(shift) = %then %let shift = 0;
%sysfunc(intnx(month, &Y., &shift. , Same), &ofmt.)
%mend monthShift;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Shift one up and one down:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put 
  Past:    %monthShift(2023, 1, -1) 
  Current: %monthShift(2023, 1    )
  Future:  %monthShift(2023, 1, +1)
;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Shift by expression:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let n = 2;
  %put 
    %monthShift(2023, 1, +1 + &n.*3)
  ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Shift with default values:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put %monthShift();
  %put %monthShift(2023);
  %put %monthShift(2023,Jan);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Shift with months names:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put 
    %monthShift(2023,Jan,-1)
    %monthShift(2023,Feb,-2)
    %monthShift(2023,Mar,-3)
    %monthShift(2023,Apr,-4)
    %monthShift(2023,May,-5)
    %monthShift(2023,Jun,-6)
    %monthShift(2023,Jul,-7)
    %monthShift(2023,Aug,-8)
    %monthShift(2023,Sep,-9)
    %monthShift(2023,Oct,-10)
    %monthShift(2023,Nov,-11)
    %monthShift(2023,Dec,-12)
  ;

  %put 
    %monthShift(2023,January,12)
    %monthShift(2023,February,11)
    %monthShift(2023,March,10)
    %monthShift(2023,April,9)
    %monthShift(2023,May,8)
    %monthShift(2023,June,7)
    %monthShift(2023,July,6)
    %monthShift(2023,August,5)
    %monthShift(2023,September,4)
    %monthShift(2023,October,3)
    %monthShift(2023,November,2)
    %monthShift(2023,December,1)
  ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** Play with formatting:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %put
    %monthShift(2023, 1, +1                )
    %monthShift(2023, 1, +1, ofmt=yymm7. )
    %monthShift(2023, 1, +1, ofmt=yymmd7.)
    %monthShift(2023, 1, +1, ofmt=yymms7.)
  ;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6.** Read monthly data with `noDSNFERR` option:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  data
    A202210 A202211 A202212
    A202301 A202302 A202303
    A202304 A202305 A202306
    A202307 A202308 A202309
  ;
    set sashelp.class;
  run;


  options noDSNFERR; 
  data ALL;
    set 
      A%monthShift(2023, 9, -12)  - A%monthShift(2023, 9)
    ;
  run;
  options DSNFERR;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since 2023.                           */
/*                                                                     */
/*  Code is under the MIT license. If you want - you can use it.       */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/


