/*** HELP START ***//*
 
## >>> `%createDHArray()` macro: <<< <a name="createdharray-macro"></a> #######################

The `%createDHArray()` macro allows to generate
a `dynamic hash array` which is a FCMP based approach 
to create *dynamically* allocated **numeric** 
or **character** array

*Note:* Arrays provided by the macro are *one dimensional* arrays.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%createDHArray( 
  arrayName           
 <,debug=0>
 <,type=8>
 <,outlib=work.DFAfcmp.package>
 <,hashexp=13>
 <,header=1>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `arrayName`    - *Required*, creates a FCMP call subroutine which is also 
                    an array name. In the data step it is used in form of 
                    a call subroutine, e.g. `call arrayName("Allocate", -3, 3)`.
                    Has to satisfy FCMP function naming requirements, but with 
                    maximum of 24 characters.

* `debug=`        - *Optional*, the default value is `0`.
                    If set to `1` then it turns on a debugging mode.

* `type=`         - *Optional*, the default value is `8`.
                    Indicates what *type* (numeric/character) and *length* 
                    are data portion of generated array. Should be in line 
                    with the LENGTH statement, e.g. `8`, `$ 30`, etc.
                    Determines if the `value` argument is numeric or character.

* `outlib=`       - *Optional*, the default value is `work.DFAfcmp.package`.
                    It points the default location for new generated dynamic 
                    function arrays compiled by FCMP.
                    *Hint!* Keep it as it is.

* `hashexp=`      - *Optional*, the default value is `13`. It is the default `hashexp=` 
                    value for internal hash table used by the function.

* `header=`       - *Optional*, the default value is `1`. Indicates if 
                    the `proc fcmp outlib = &outlib.;` header is added to 
                    the executed code. If not 1 then no header is added.

**Created function arguments description**:

A function generated by the macro is:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
call &arrayName.(IO, position, value)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
and accepts the following list of arguments and values:

1. `IO` - is a *character* steering argument, possible 
          values and behaviour they call are the following:
  - `O`, `Output`, `R`, `Return`    - to get the data from an array,
  - `I`, `Input`                    - to insert the data into an array,
  - `C`, `Clear`                    - to reduce an array to a single empty cell,
  - `L`, `Low`, `Lower`, `Lbound`   - to return minimal position of index,
  - `H`, `High`, `Higher`, `Hbound` - to return maximal position of index.
                
2. `position` - is a *numeric* argument and depends on the `IO` value. 
                Behaves in the following way:
  - for `O`, `Output`, `R`, `Return`/ `I`, and `Input` it is an array 
    index from (into) which data is get (put),
  - for `C` it is ignored,
  - for `L`, `Low`, `Lower`, and `Lbound` it returns the first position of an index,
  - for `H`, `High`, `Higher`, and `Hbound` it returns the last position of an index,
  - otherwise is not modified.
                
3. `value` - is a *numeric* or *character* argument (determined by the `type=`) 
             and depends on the `IO` value. Behaves in the following way:
  - for `O`, `Output`, `R`, and `Return` it holds value retrieved from an array from a given position,
  - for `I`, `Input` it holds the value inserted into an array into a given position,
  - for `C` is ignored,
  - for `L`, `Low`, `Lower`, and `Lbound` returns first value of index,
  - for `H`, `High`, `Higher`, and `Hbound` returns last value of index,
  - otherwise is not modified.

The `position` and the `value` arguments are **outargs**, i.e. can be changed by the function.

*//*** HELP END ***/

/**************/
%macro createDHArray(/* macro Create Dynamic Function Array */
  arrayName                      /* array name, in datastep used in form of call subroutine
                                    e.g. call arrayName("Input", -3, 3) 
                                  */
, debug=0                        /* if 1 then turns on debugging mode */
, type=8                         /* type of array data portion, should be in 
                                    line with LENGTH statement, e.g. 8, $ 12, etc. 
                                  */
, outlib = work.DFAfcmp.package  /* default location for compiled functions */
, hashexp=13                     /* default hashexp value for hash table */
, header=1                       /* adding Proc FCMP header to the executed code 
                                    if other than 1 then _no_ "proc fcmp outlib = &outlib.;"
                                    will be added. 
                                  */
);
%if %superq(header) = 1 %then
%do;
  proc fcmp outlib = &outlib.;
%end;
  subroutine &arrayName.(
      IO $     /* CHARACTER
                * steering argument:
                * O, Output, R, Return    - get the data from an array
                * I, Input                - insert the data into an array
                * C, Clear                - reduce an array to a single empty cell
                * L, Low, Lower, Lbound   - return minimal position of index
                * H, High, Higher, Hbound - return maximal position of index
                */
    , position /* NUMERIC
                * for O, Output, R, Return/ I, Input it is an array's index from(into) which data is get(put)
                * for C ignored
                * for L, Low, Lower, Lbound returns first position of index
                * for H, High, Higher, Hbound returns last position of index
                * otherwise does not modify value
                */
    , value %qsysfunc(compress(&type., $, k)) 
               /* NUMERIC/CHARACTER  
                * for O, Output, R, Return it holds value retrieved from an array on a given position
                * for I, Input it holds the value inserted into an array on a given position
                * for C ignored
                * for L, Low, Lower, Lbound returns first value of index
                * for H, High, Higher, Hbound returns last value of index
                * otherwise does not modify value
                */
    );
    outargs position, value;

/**************/

    length position 8 value &type.;
    declare hash H(ordered:"A", duplicate:"R", hashexp:&hashexp.);
    _RC_ = H.defineKey("position");
    _RC_ = H.defineData("position");
    _RC_ = H.defineData("value");
    _RC_ = H.defineDone();
    declare hiter I("H");
  
    select(upcase(IO));    
    /* Output - get the data from an array */
      when ('O', 'OUTPUT', 'R', 'RETURN')
      do;
        if H.find() then call missing(value);
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&arrayName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
        %end;
        return;
      end;
    
    /* Input - insert the data into an array */
      when ('I', 'INPUT'
       %if %qsysfunc(compress(&type., $, k)) = %then
        %do;
         ,'+', 'ADD'
        %end;
      )
      do; 

        %if %qsysfunc(compress(&type., $, k)) = %then
        %do;
        if upcase(IO) = "+" or upcase(IO) = "ADD" then
          do;
            _TO_BE_MODIF_VALUE_ = value;
            _RC_ = H.find();
            if value > .z then 
              value = _TO_BE_MODIF_VALUE_ + value;
            else value = _TO_BE_MODIF_VALUE_;
          end;
        %end;
       
        _RC_ = H.replace();
        %if &debug %then 
        %do;
          _T_ = H.num_items();
          put "NOTE:[&arrayName.] Debug" "dim(TEMP)=" _T_ "value=" value "position=" position;
        %end;
        return;
      end;

    /* Clear - reduce an array to a single empty cell */
      when ('C', 'CLEAR')
      do;
         _RC_ = H.clear();
        return;
      end;

    /* Dimension - returns minimal and maximal index 
     */
      when ('L', 'LOW', 'LOWER', 'LBOUND')
      do;
        _RC_ = I.first();
        _RC_ = I.prev();
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&arrayName.] Debug" "dim(TEMP)=" _T_;
        %end;
        return;
      end;

      when ('H', 'HIGH', 'HIGHER', 'HBOUND')
      do;
        _RC_ = I.last();
        _RC_ = I.next();
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&arrayName.] Debug" "dim(TEMP)=" _T_;
        %end;
        return;
      end;

      otherwise;
    end;

    put "WARNING: IO parameter value" IO "is unknown.";
    put "NOTE: Use: 'O', 'OUTPUT', 'R', 'RETURN', 'I', 'INPUT', 'C', 'CLEAR'";
    put "NOTE:  or  'L', 'LOW', 'LOWER', 'LBOUND', 'H', 'HIGH', 'HIGHER', 'HBOUND'";
    return;
  endsub;

%if %superq(header) = 1 %then
%do;
  run;
%end;
/*QUIT;*/
%mend createDHArray;



/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Dynamic, Hash-based, and Character array:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHArray(ArrDHC, type = $ 12); 
  options APPEND=(cmplib = WORK.DFAfcmp) ; 
 
  %let zeros = 6; *[to test bigger sizes];
  data Example1; 
   
    t = time(); 
    do _I_ = -1e&zeros. to 1e&zeros.; 
      _X_ = put(_I_*10, z12.); 
      call ArrDHC("Input", _I_, _X_); 
    end; 
    t = time() - t; 
    put t= / _X_= /; 
   
 * get the size info ; 
    LB = 0; HB = 0; 
    drop LB HB; 
      call ArrDHC('Lbound', LB, _X_); 
      call ArrDHC('Hbound', HB, _X_); 
    put LB= HB= /; 
   
    t = time(); 
    do _I_ = HB + 1 to LB - 1 by -1; 
      call ArrDHC('Output', _I_, _X_); 
      output;  
    end; 
    t = time() - t; 
    put t= / _X_= /; 
    
 * clear for further reuse ; 
    call ArrDHC('C', ., ''); 
    
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Dynamic, Hash-based, and Numeric array:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHArray(ArrDHN);
  options APPEND=(cmplib = WORK.DFAfcmp) ;

  data Example2;
  
    do i = -2 to 2;
      call ArrDHN("Input", i, 2**i);
    end;

    do i = -2 to 2;
      call ArrDHN("+", i, -10);
    end;
    
    v = .;
    do i = -2 to 2;
      call ArrDHN("Output", i, v);
      put i= v=;
    end;

  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since July 2019.                      */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
