/*** HELP START ***//*
 
## >>> `%createDHFifo()` macro: <<< <a name="createdhfifo-macro"></a> #######################

The `%createDHFifo()`  macro allows to generate
a `dynamic hash fifo` which is a FCMP based approach 
to create dynamically allocated numeric or character 
"first in first out" [queue](https://en.wikipedia.org/wiki/FIFO_(computing_and_electronics))

Interesting reading about implementing a fifo via hash table
can be found in *chapter 10.4* of the:
*"Data Management Solutions Using SAS Hash Table Operations: 
  A Business Intelligence Case Study"* book
by Paul Dorfman and Don Henderson.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%createDHFifo( 
  fifoName           
 <,debug=0>
 <,type=8>
 <,outlib=work.DFAfcmp.package>
 <,hashexp=13>
 <,header=1>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `fifoName`     - *Required*, creates a FCMP call subroutine which is also 
                    a fifo name. In the data step it is used in form of 
                    a call subroutine, e.g. `call fifoName("Enqueue", 3)`.
                    Has to satisfy FCMP function naming requirements, but with 
                    maximum of 24 characters.

* `debug=`        - *Optional*, the default value is `0`.
                    If set to `1` then it turns on a debugging mode.

* `type=`         - *Optional*, the default value is `8`.
                    Indicates what *type* (numeric/character) and *length* 
                    are data portion of generated array. Should be in line 
                    with the LENGTH statement, e.g. `8`, `$ 30`, etc.
                    Determines if the `value` argument is numeric or character.

* `outlib=`       - *Optional*, the default value is `work.DFAfcmp.package`.
                    It points the default location for new generated dynamic 
                    function arrays compiled by FCMP.
                    *Hint!* Keep it as it is.

* `hashexp=`      - *Optional*, the default value is `13`. It is the default `hashexp=` 
                    value for internal hash table used by the function.

* `header=`       - *Optional*, the default value is `1`. Indicates if 
                    the `proc fcmp outlib = &outlib.;` header is added to 
                    the executed code. If not 1 then no header is added.

**Created function arguments description**:

A function generated by the macro is:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
call &fifoName.(IO, value)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
and accepts the following list of arguments and values:

1. `IO` - is a *character* steering argument, possible 
          values and behaviour they call are the following:
  - `O`, `Output`, `D`, `Dequeue`, `R`, `Return` - to get the data from a fifo (and remove it from the fifo)
  - `I`, `Input`, `E`, `Enqueue`, and `Insert`   - to insert the data into a fifo
  - `C`, `Clear`                                 - to reduce a fifo to an empty one
  - `P`, `Peek`, `T`, and `Tail`                 - to peek the data from a fifo (and NOT remove it from the fifo)
  - `H`, `Head`                                  - to peek the data from a fifo head (and NOT remove it from the fifo)
  - `Sum`                      - returns sum of nonmissing numeric elements of a stack
  - `Avg`, `Mean`, `Average`   - returns average of nonmissing numeric elements of a stack
  - `Nonmiss`, `Cnt`           - returns number of nonmissing elements of a stack
  - `Height`                   - returns height a stack
  
2. `value` - is a *numeric* or *character* argument (determined by the `type=`) 
             and depends on the `IO` value. Behaves in the following way:
  - for `O`, `Output`, `D`, `Dequeue`, `R`, `Return` it holds the value popped from the fifo,
  - for `I`, `Input`, `E`, `Enqueue`, `Insert` it holds the value to be pushed into the fifo,
  - for `C`, `Clear` it is ignored,
  - for `P`, `Peek` holds the value peeked from the fifo,
  - for `Sum`, `Nonmiss`, `Cnt`, `Avg`, `Mean`, `Average`, and `Height` it returns calculated summary value.

The `value` argument is **outarg**, i.e. can be changed by the function.

*//*** HELP END ***/

/**************/
%macro createDHFifo( /* macro Create Dynamic Function Fifo Queue */
  fifoName                       /* fifo name, in datastep used in form of call subroutine
                                    e.g. call fifoName("Enqueue", 3) 
                                  */
, debug=0                        /* if 1 then turns on debugging mode */
, type=8                         /* type of fifo data portion, should be in 
                                    line with LENGTH statement, e.g. 8, $ 12, etc. 
                                  */
, outlib = work.DFAfcmp.package  /* default location for compiled functions */
, hashexp=13                     /* default hashexp value for hash table */
, header=1                       /* adding Proc FCMP header to the executed code 
                                    if other than 1 then _no_ "proc fcmp outlib = &outlib.;"
                                    will be added. 
                                  */
);
%if %superq(header) = 1 %then
%do;
  proc fcmp outlib = &outlib.;
%end;
  subroutine &fifoName.(
      IO $     /* CHARACTER
                * steering argument:
                * O, Output, D, Dequeue, R, Return  - gets the data from a stack  
                *                                     and removes it from the top 
                * I, Input, E, Enqueue, Insert      - inserts the data into a stack
                * C, Clear                          - reduces a fifo to an empty one
                * P, Peek, T, Tail                  - peeks the data from a fifo tail
                *                                     and NOT removes it from the top
                * H, Head                           - peeks the data from a fifo head
                *                                     and NOT removes it from the top  
                *
                * Sum                  - returns sum of nonmissing numeric elements of a stack
                * Avg, Mean, Average   - returns average of nonmissing numeric elements of a stack
                * Nonmiss, Cnt         - returns number of nonmissing elements of a stack
                * Height               - returns height a stack
                */
    , value %qsysfunc(compress(&type., $, k)) 
               /* NUMERIC/CHARACTER  
                * for O, Output, D, Dequeue, R, Return holds a value popped from a fifo
                * for I, Input, E, Enqueue, Insert holds a value to be pushed into a fifo
                * for C, Clear ignored
                * for P, Peek holds a value peeked from a fifo
                * for Sum, Nonmiss, Cnt, Avg, Mean, Average, Height returns calculated summary value
                */
    );
    outargs value;

/**************/

    length position 8 value &type.;
    declare hash H(ordered:"A", duplicate:"R", hashexp:&hashexp.);
    _RC_ = H.defineKey("position");
    _RC_ = H.defineData("position");
    _RC_ = H.defineData("value");
    _RC_ = H.defineDone();
    declare hiter I("H"); 

    static _sum_ .;
    static _cnt_ .;
    /* TODO: figure out smart way for min and max */
    /*
    static _min_ .;
    static _max_ .;
    */

    select(upcase(IO));  
      /* Output - get the data from a fifo */
      when ('O', 'OUTPUT', 'D', 'DEQUEUE', 'R', 'RETURN')
        do;
          call missing(value);
          _RC_ = I.first();
          _RC_ = I.prev();

          %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
            %do; 
              /* since value is a character type then do nothing */
              _cnt_ = sum(_cnt_, -(value ne " "));
            %end;
          %else /* numeric type */
            %do;
              _sum_ = sum(_sum_, -(value));
              _cnt_ = sum(_cnt_, -(value > .z));
            %end;

          _RC_ = H.remove();

          %if &debug %then %do;
            _T_ = H.num_items();
            put "NOTE:[&stackName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
          %end;
          return;
        end;
      
      /* Input - insert the data into a stack */
      when ('I', 'INPUT', 'E', 'ENQUEUE' ,'INSERT')
        do;   
          
          position = H.num_items() + 1;
          _RC_ = H.replace();

          %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
            %do; 
              /* since value is a character type then do nothing */
              _cnt_ = sum(_cnt_, (value ne " "));
            %end;
          %else /* numeric type */
            %do; 
              _sum_ = sum(_sum_, (value));
              _cnt_ = sum(_cnt_, (value > .z));
            %end;

          %if &debug %then %do;
            _T_ = H.num_items();
            put "NOTE:[&stackName.] Debug" "dim(TEMP)=" _T_ "value=" value "position=" position;
          %end;
          return;
        end;

      /* Tail - peeks the data from a fifo tail without removing */
      when ('P', 'PEEK', 'T', 'TAIL')
        do;
          call missing(value);
          _RC_ = I.first();
          _RC_ = I.prev();
          %if &debug %then %do;
            _T_ = H.num_items();
            put "NOTE:[&stackName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
          %end;
          return;
        end;

      /* Head - peeks the data from a fifo head without removing */
      when ('H', 'HEAD')
        do;
          call missing(value);
          _RC_ = I.last();
          _RC_ = I.next();
          %if &debug %then %do;
            _T_ = H.num_items();
            put "NOTE:[&stackName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
          %end;
          return;
        end;

      /* Clear - reduce a stack to a empty one */
      when ('C', 'CLEAR')
        do;
          _RC_ = H.clear();
          _sum_ = .;
          _cnt_ = .;
          return;
        end;

      /* Statistic - returns selected statistic */
      %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
        %do; 
          when ('NONMISS', 'CNT')
            do;
              value = strip(put(_cnt_, best32.));
              return;
            end;
          when ('HEIGHT')
            do;
              value = strip(put(H.num_items(), best32.));
              return;
            end;             
        %end;
      %else /* numeric type */
        %do; 
          when ('SUM') 
            do;
              value = _sum_; /* Sum */
              return;
            end;
          when ('AVG', 'MEAN', 'AVERAGE') 
            do;
              value = divide(_sum_, _cnt_); /* Average */
              return;
            end;         
          when ('NONMISS', 'CNT')
            do;
              value = _cnt_; /* NonMiss */
              return;
            end;
          when ('HEIGHT')
            do;
              value = H.num_items(); /* StackHeight */
              return;
            end;
        %end;

      otherwise;
    end;

    put "WARNING: IO parameter value" IO "is unknown.";
    put "NOTE: Use: 'O', 'OUTPUT', 'D', 'DEQUEUE', 'R', 'RETURN'";
    put "NOTE:  or  'I', 'INPUT', 'E', 'ENQUEUE' ,'INSERT'";
    put "NOTE:  or  'P', 'PEEK', 'T', 'TAIL', 'H', 'HEAD', 'C', 'CLEAR'";
    put "NOTE:  or  'SUM', 'AVG', 'MEAN', 'AVERAGE', 'NONMISS', 'CNT', 'HEIGHT'";
    return;
  endsub;

%if %superq(header) = 1 %then
%do;
  run;
%end;
%mend createDHFifo;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Dynamic, Hash-based, and Character fifo:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHFifo(FifoDHC, type = $ 12); 
  options APPEND=(cmplib = WORK.DFAfcmp) ; 

 
  %let zeros = 6; *[to test bigger sizes];
  data Example1;
 
    t = time(); drop t; 
    do _I_ = 1 to 1e&zeros.; 
      _X_ = put(_I_*10, z12.); 
      call FifoDHC("Enqueue", _X_); 
    end; 
    t = time() - t;
 
    call FifoDHC("Height", _X_); 
    put t= / _X_=; 
   
    t = time(); 
    do _I_ = 1 to 1e&zeros. + 3; 
      call FifoDHC('Dequeue', _X_); 
      output;  
    end; 
    t = time() - t; 

    call FifoDHC("Height", _X_); 
    put t= / _X_=; 
 
    %* clear for further reuse *; 
    call FifoDHC('Clear', '');  

  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Dynamic, Hash-based, and Numeric fifo:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHFifo(FifoDHN); 
  options APPEND=(cmplib = WORK.DFAfcmp) ; 

  data Example2;
 
    do _I_ = 1,.,2,.,3,.,4,.,5,.,6;  
      call FifoDHN("E", _I_); 
    end; 

    call FifoDHN("Sum", _I_); 
    put "Sum    " _I_=;

    call FifoDHN("Avg", _I_); 
    put "Avg    " _I_=;

    call FifoDHN("Cnt", _I_); 
    put "Cnt    " _I_=;
   
    call FifoDHN("Height", _I_); 
    put "Height " _I_=; 

    call FifoDHN("Tail", _I_); 
    put "Tail of fifo is " _I_=; 

    call FifoDHN("Height", _I_); 
    put "Height after Tail " _I_=; 

    call FifoDHN("Head", _I_); 
    put "Head of fifo is " _I_=; 

    call FifoDHN("Height", _I_); 
    put "Height after Head" _I_=; 

    _X_ = 0;
    do _I_ = 1 to _I_; 
      call FifoDHN('D', _X_); 
      output;  
    end; 

    call FifoDHN("Height", _I_); 
    put "Height " _I_=;

  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since July 2019.                      */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
