/*** HELP START ***//*
 
## >>> `%createDHOrdStack()` macro: <<< <a name="createdhordstack-macro"></a> #######################

The `%createDHOrdStack()` macro allows to generate
a `dynamic ORDERED hash stack` which is a FCMP based approach 
to create dynamically allocated numeric or character 
*ordered* [stack](https://en.wikipedia.org/wiki/Stack_(abstract_data_type))

Interesting reading about implementing a stack via hash table
can be found in *chapter 10.4* of the:
*"Data Management Solutions Using SAS Hash Table Operations: 
  A Business Intelligence Case Study"* book
by Paul Dorfman and Don Henderson.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%createDHOrdStack( 
  fifoName           
 <,debug=0>
 <,type=8>
 <,order=A>
 <,outlib=work.DFAfcmp.package>
 <,hashexp=13>
 <,header=1>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `stackName`    - *Required*, creates a FCMP call subroutine which is also 
                    a stack name. In the data step it is used in form of 
                    a call subroutine, e.g. `call stackName("Push", 3)`.
                    Has to satisfy FCMP function naming requirements, but with 
                    maximum of 24 characters.

* `debug=`        - *Optional*, the default value is `0`.
                    If set to `1` then it turns on a debugging mode.

* `type=`         - *Optional*, the default value is `8`.
                    Indicates what *type* (numeric/character) and *length* 
                    are data portion of generated array. Should be in line 
                    with the LENGTH statement, e.g. `8`, `$ 30`, etc.
                    Determines if the `value` argument is numeric or character.

* `order=`        - *Optional*, the default value is `A`.
                    Indicates a method of ordering of the stack,
                    allowed values are: `A` for ascending and `D` for descending.

* `outlib=`       - *Optional*, the default value is `work.DFAfcmp.package`.
                    It points the default location for new generated dynamic 
                    function arrays compiled by FCMP.
                    *Hint!* Keep it as it is.

* `hashexp=`      - *Optional*, the default value is `13`. It is the default `hashexp=` 
                    value for internal hash table used by the function.

* `header=`       - *Optional*, the default value is `1`. Indicates if 
                    the `proc fcmp outlib = &outlib.;` header is added to 
                    the executed code. If not 1 then no header is added.

**Created function arguments description**:

A function generated by the macro is:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
call &stackName.(IO, value)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
and accepts the following list of arguments and values:

1. `IO` - is a *character* steering argument, possible 
          values and behaviour they call are the following:
  - `O`, `Output`, `Pop`, `G`, `Get`, `R`, `Return` - to get the data from a stack (and remove it from the top),
  - `I`, `Input`, `Push`, `Put`, `Insert`           - to insert the data into a stack,
  - `C`, `Clear`                                    - to reduce a stack to an empty one,
  - `P`, `Peek`                                     - to peek the data from a stack (and NOT remove it from the top),
  - `Sum`                    - returns sum of non-missing numeric elements of a stack,
  - `Avg`, `Mean`, `Average` - returns average of non-missing numeric elements of a stack,
  - `Nonmiss`, `Cnt`, `Nnm`  - returns number of non-missing elements of a stack,
  - `Height`                 - returns height a stack,
  - `Min`, `Minimum`         - returns minimum of non-missing elements of a stack,
  - `Max`, `Maximum`         - returns maximum of non-missing elements of a stack.
  
2. `value` - is a *numeric* or *character* argument (determined by the `type=`) 
             and depends on the `IO` value. Behaves in the following way:
  - for `O`, `Output`, `Pop`, `G`, `Get`, `R`, `Return` it holds a value popped from a stack,
  - for `I`, `Input`, `Push`, `Put`, `Insert` it holds a value to be pushed into a stack,
  - for `C`, `Clear` it is ignored,
  - for `P`, `Peek` it holds a value peeked from a stack,
  - for `Sum`, `Nonmiss`, `Cnt`, `Avg`, `Mean`, `Average`, `Height`, `Min`, `Minimum`, `Max`, and `Maximum` 
    it returns calculated summary value,

The `value` argument is **outarg**, i.e. can be changed by the function.

*//*** HELP END ***/

/**************/
%macro createDHOrdStack( /* macro Create Dynamic Function Ordered Stack */
  stackName                      /* stack name, in datastep used in form of call subroutine
                                    e.g. call stackName("Push", 3) 
                                  */
, debug=0                        /* if 1 then turns on debugging mode */
, type=8                         /* type of stack data portion, should be in 
                                    line with LENGTH statement, e.g. 8, $ 12, etc. 
                                  */
, order=A                        /* method of ordering of the stack,
                                    allowed values are: A, Ascending or D, Descending
                                  */
, outlib = work.DFAfcmp.package  /* default location for compiled functions */
, hashexp=13                     /* default hashexp value for hash table */
, header=1                       /* adding Proc FCMP header to the executed code 
                                    if other than 1 then _no_ "proc fcmp outlib = &outlib.;"
                                    will be added. 
                                  */
);
%if %superq(header) = 1 %then
%do;
  proc fcmp outlib = &outlib.;
%end;
  subroutine &stackName.(
      IO $     /* CHARACTER
                * steering argument:
                * O, Output, Pop, G, Get, R, Return - gets the data from a stack  
                *                                     and removes it from the top 
                * I, Input, Push, Put ,Insert       - inserts the data into a stack
                * C, Clear                          - reduces a stack to an empty one
                * P, Peek                           - peeks the data from a stack 
                *                                     and NOT removes it from the top 
                *
                * Sum                  - returns sum of nonmissing numeric elements of a stack
                * Avg, Mean, Average   - returns average of nonmissing numeric elements of a stack
                * Nonmiss, Cnt, Nnm    - returns number of nonmissing elements of a stack
                * Height               - returns height a stack
                * Min, Minimum         - returns minimum of nonmissing elements of a stack
                * Max, Maximum         - returns minimum of nonmissing elements of a stack
                */
    , value %qsysfunc(compress(&type., $, k)) 
               /* NUMERIC/CHARACTER  
                * for O, Output, Pop, G, Get, R, Return holds a value popped from a stack
                * for I, Input, Push, Put ,Insert holds a value to be pushed into a stack
                * for C, Clear ignored
                * for P, Peek holds a value peeked from a stack
                * for Sum, Nonmiss, Cnt, Avg, Mean, Average, Height, Min, Minimum, Max, Maximum 
                *     returns calculated summary value
                */
    );
    outargs value;

/**************/

    length position positionTMP 8 value &type.;
    static position 0;
    declare hash H(ordered:"&order.", duplicate:"R", hashexp:&hashexp.);
    _RC_ = H.defineKey("value");
    _RC_ = H.defineKey("position");
    _RC_ = H.defineDone();
    declare hiter I("H"); 

    static _sum_ .;
    static _cnt_ .;
 
    select(upcase(IO));  
      /* Output - get the data from a stack */
      when ('O', 'OUTPUT', 'POP', 'G', 'GET', 'R', 'RETURN')
        do;
          positionTMP = position;
          call missing(value,position);
          
          _RC_ = I.first();
          _RC_ = I.prev();

          %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
            %do; 
              /* since value is a character type then do nothing */
              _cnt_ = sum(_cnt_, -(value ne " "));
            %end;
          %else /* numeric type */
            %do;
              _sum_ = sum(_sum_, -(value));
              _cnt_ = sum(_cnt_, -(value > .z));
            %end;        

          _RC_ = H.remove();
          position = positionTMP;

          %if &debug %then %do;
            _T_ = H.num_items();
            put "NOTE:[&stackName.] Debug O:" "dim(TEMP)=" _T_ "value=" value "position=" position;
          %end;
          return;
        end;
    
    /* Input - insert the data into a stack */
    when ('I', 'INPUT', 'PUSH', 'PUT' ,'INSERT')
      do;      
        position = position + 1;
        _RC_ = H.replace();

        %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
          %do; 
            /* since value is a character type then do nothing */
            _cnt_ = sum(_cnt_, (value ne " "));
          %end;
        %else /* numeric type */
          %do; 
            _sum_ = sum(_sum_, (value));
            _cnt_ = sum(_cnt_, (value > .z));
          %end;

        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&stackName.] Debug I:" "dim(TEMP)=" _T_ "value=" value "position=" position;
        %end;
        return;
      end;

    /* Peek - peeks the data from a stack without removing */
    when ('P', 'PEEK')
      do;
        call missing(value);
        _RC_ = I.first();
        _RC_ = I.prev();
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&stackName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
        %end;
        return;
      end;


      /* Clear - reduce a stack to a empty one */
      when ('C', 'CLEAR')
        do;
          _RC_ = H.clear();
          position = 0;
          positionTMP = 0;
          _sum_ = .;
          _cnt_ = .;
          return;
        end;

      /* Statistic - returns selected statistic */
      %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
        %do; 
          when ('NONMISS', 'CNT')
            do;
              value = strip(put(_cnt_, best32.));
              return;
            end;
          when ('HEIGHT')
            do;
              value = strip(put(H.num_items(), best32.));
              return;
            end;             
        %end;
      %else /* numeric type */
        %do; 
          when ('SUM') 
            do;
              value = _sum_; /* Sum */
              return;
            end;
          when ('AVG', 'MEAN', 'AVERAGE') 
            do;
              value = divide(_sum_, _cnt_); /* Average */
              return;
            end;         
          when ('NNM', 'NONMISS', 'CNT')
            do;
              value = _cnt_; /* NonMiss */
              return;
            end;
          when ('HEIGHT')
            do;
              value = H.num_items(); /* StackHeight */
              return;
            end;
          when ('MIN', 'MINIMUM') 
            do; /* Min */
              %if %qupcase(&order.)=A %then 
                %do;
                  value = .;
                  _RC_ = I.first();
                  do _N_ = 1 to H.num_items() while (not (value > .z));
                    _RC_ = I.next();
                  end;
                %end;
              %else %if %qupcase(&order.)=D %then
                %do;
                  value = .;
                  _RC_ = I.last();
                  do _N_ = 1 to H.num_items() while (not (value > .z));
                    _RC_ = I.prev();
                  end;
                %end;
              return;
            end;
          when ('MAX', 'MAXIMUM') 
            do; /* Max */
              %if %qupcase(&order.)=D %then 
                %do;
                  value = .;
                  _RC_ = I.first();
                  _RC_ = I.prev();
                %end;
              %else %if %qupcase(&order.)=A %then
                %do;
                  value = .;
                  _RC_ = I.last();
                  _RC_ = I.next();
                %end;
              return;
            end;
        %end;

      otherwise;
    end;

    put "WARNING: IO parameter value" IO "is unknown.";
    put "NOTE: Use: 'O', 'OUTPUT', 'POP', 'G', 'GET', 'R', 'RETURN'";
    put "NOTE:  or  'I', 'INPUT', 'PUSH', 'PUT' ,'INSERT'";
    put "NOTE:  or  'P', 'PEEK', 'C', 'CLEAR'";
    put "NOTE:  or  'SUM', 'AVG', 'MEAN', 'AVERAGE', 'NNM', 'NONMISS', 'CNT', 'HEIGHT'";
    put "NOTE:  or  'MIN', 'MINIMUM', 'MAX', 'MAXIMUM'";
    return;
  endsub;

%if %superq(header) = 1 %then
%do;
  run;
%end;
%mend createDHOrdStack;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Dynamic, Hash-based, and Character Descending Ordered stack:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHOrdStack(DescStackC, type = $ 12, order=D); 
  options APPEND=(cmplib = WORK.DFAfcmp) ;
 
  data Example1; 
    
    do _X_ = "A","B"," ","C","A"," ","B","C"; 
      call DescStackC("Push", _X_); 
    end; 
  
    length S $ 12;
    call DescStackC('Height', S); 
    put 'Height ' S;

    do until(strip(S) = "0"); 
      call DescStackC('Get', _X_); 
      call DescStackC('Height', S);
      put S= _X_=;
      output;
    end; 
    
    %* clear for further reuse *; 
    call DescStackC('Clear',''); 

  run; 
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Dynamic, Hash-based, and Numeric Ascending Ordered stack:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHOrdStack(DescStackN, order=A); 
  options APPEND=(cmplib = WORK.DFAfcmp) ;
 
  data Example2; 
   
    call missing(Sum, Avg, Min, Max, Cnt, Hgt, Peek);
    do _X_ = 1,6,2,.,5,3,4; 
      call DescStackN("Put", _X_); 
      call DescStackN('Sum', Sum);
      call DescStackN('Avg', Avg);
      call DescStackN('Min', Min);
      call DescStackN('Max', Max);
      call DescStackN('Cnt', Cnt);
      call DescStackN('Height', Hgt);
      put (_ALL_) (=);  
    end; 
   
    call DescStackN('Peek', Peek); 
    put Peek=;  

    do _I_ = 1 to Hgt; 
      call DescStackN('Output', _X_);
      keep _X_; 
      if _X_ > .z then output; 
    end; 
   
    call DescStackN('Peek', Peek); 
    put Peek=;
   
    %* clear for further reuse *; 
    call DescStackN('Clear',.);  

  run; 
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, July 2019.                            */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
