/* dynamicpriorityqueuebyfunctionhash is a FCMP based approach to create 
   dynamically allocated priority queue 
*/

/*** HELP START ***//*
 
## >>> `%createDHPrtQueue()` macro: <<< <a name="createdhprtqueue-macro"></a> #######################


The `%createDHPrtQueue()` macro allows to generate
a `dynamic PRIORITY hash queue` which is a FCMP based approach 
to create dynamically allocated numeric or character 
[priority queue](https://en.wikipedia.org/wiki/Priority_queue)

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%createDHPrtQueue( 
  fifoName           
 <,debug=0>
 <,type=8>
 <,newOnTop=+>
 <,outlib=work.DFAfcmp.package>
 <,hashexp=13>
 <,header=1>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `queueName`    - *Required*, creates a FCMP call subroutine which is also 
                    a queue name. In the data step it is used in form of 
                    a call subroutine, e.g. `call queueName("Bottom", 1, 3)`.
                    Has to satisfy FCMP function naming requirements, but with 
                    maximum of 24 characters.

* `debug=`        - *Optional*, the default value is `0`.
                    If set to `1` then it turns on a debugging mode.

* `type=`         - *Optional*, the default value is `8`.
                    Indicates what *type* (numeric/character) and *length* 
                    are data portion of generated array. Should be in line 
                    with the LENGTH statement, e.g. `8`, `$ 30`, etc.
                    Determines if the `value` argument is numeric or character.

* `newOnTop=`     - *Optional*, the default value is `+`.
                    Indicates how to keep order in the same priority group,
                    allowed values are `+` or `-`. Plus(`+`) sets new elements
                    at the top of the group, minus(`-`) at the bottom.

* `outlib=`       - *Optional*, the default value is `work.DFAfcmp.package`.
                    It points the default location for new generated dynamic 
                    function arrays compiled by FCMP.
                    *Hint!* Keep it as it is.

* `hashexp=`      - *Optional*, the default value is `13`. It is the default `hashexp=` 
                    value for internal hash table used by the function.

* `header=`       - *Optional*, the default value is `1`. Indicates if 
                    the `proc fcmp outlib = &outlib.;` header is added to 
                    the executed code. If not 1 then no header is added.

**Created function arguments description**:

A function generated by the macro is:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
call &queueName.(IO, position, value)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
and accepts the following list of arguments and values:

1. `IO` - is a *character* steering argument, possible 
          values and behaviour they call are the following:
  - `O`, `Output`, `D`, `Dequeue`, `R`, `Return` - it pops/gets/outputs the data from the queue head (high priority),
  - `B`, `Bottom`                                - it pops/gets/outputs the data from the queue tail (low priority),
  - `I`, `Input`, `E`, `Enqueue`, `Insert`       - it push/puts/inserts the data into the queue,
  - `C`, `Clear`                                 - it reduces a queue to an empty one,
  - `H`, `Head`                                  - it peeks the data from the queue head and NOT removes it,
  - `T`, `Tail`                                  - it peeks the data from the queue tail and NOT removes it,
  - `Sum`                                        - it returns sum of non-missing *numeric* elements of the queue,
  - `Avg`, `Mean`, `Average`                     - it returns average of non-missing *numeric* elements of the queue,
  - `Nonmiss`, `Cnt`                             - it returns number of non-missing elements of the queue,
  - `Height`                                     - it returns height of the queue.

2. `position` - is a *numeric* argument and depends on the `IO` value. 
                Behaves in the following way:
  - for `O`, `Output`, `D`, `Dequeue`, `R`, `Return` and `B`, `Bottom`, or `H`, `Head`, `T`, `Tail` 
    it holds a priority level of value popped from the queue,
  - for `I`, `Input`, `E`, `Enqueue`, `Insert` it holds a priority level of value to be pushed into the queue,
  - for `C` ignored,
  - for *numeric* queue and `Sum`, `Nonmiss`, `Cnt`, `Avg`, `Mean`, `Average`, `Height` returns calculated summary value.

3. `value` - is a *numeric* or *character* argument (determined by the `type=`) 
             and depends on the `IO` value. Behaves in the following way:
  - for `O`, `Output`, `D`, `Dequeue`, `R`, `Return` and `B`, `Bottom` or `H`, `Head`, `T`, `Tail` 
    it holds a value popped from the queue,
  - for `I`, `Input`, `E`, `Enqueue`, `Insert` it holds a value to be pushed into the queue,
  - for `C` ignored,
  - for *numeric* queue and `Sum`, `Nonmiss`, `Cnt`, `Avg`, `Mean`, `Average`, `Height` returns calculated summary value,
  - otherwise does not modify value.

The `position` and the `value` arguments are **outargs**, i.e. can be changed by the function.

*//*** HELP END ***/

/**************/
%macro createDHPrtQueue( /* macro Create Dynamic Function Priority Queue */
  queueName                      /* queue name, in datastep used in form of call subroutine
                                    e.g. call queueName("Bottom", 1, 3) 
                                  */
, debug=0                        /* if 1 then turns on debugging mode */
, type=8                         /* type of queue data portion, should be in 
                                    line with LENGTH statement, e.g. 8, $ 12, etc. 
                                  */
, newOnTop=+                     /* indicate how to keep order in priority group,
                                    allowed values are + or -,
                                  */
, outlib = work.DFAfcmp.package  /* default location for compiled functions */
, hashexp=13                     /* default hashexp value for hash table */
, header=1                       /* adding Proc FCMP header to the executed code 
                                    if other than 1 then _no_ "proc fcmp outlib = &outlib.;"
                                    will be added. 
                                  */
);
/**************/

%if %bquote(&newOnTop.) ne %str(-) %then %let newOnTop=+;

/**************/
%if %superq(header) = 1 %then
%do;
  proc fcmp outlib = &outlib.;
%end;
  subroutine &queueName.(
      IO $     /* CHARACTER
                * steering argument:
                * O, Output, D, Dequeue, R, Return - pop/get/output the data from 
                *                                    a queue's head (high priority)
                * B, Bottom                        - pop/get/output the data from 
                *                                    a queue's tail (low priority)
                * I, Input, E, Enqueue, Insert     - push/put/insert the data into a queue
                * C, Clear                         - reduce a queue to an empty one
                * H, Head                          - peek the data from a queue's 
                *                                    head and NOT removes it
                * T, Tail                          - peek the data from a queue's 
                *                                    tali and NOT removes it
                * Sum                  - returns sum of nonmissing numeric elements of the queue
                * Avg, Mean, Average   - returns average of nonmissing numeric elements of the queue
                * Nonmiss, Cnt         - returns number of nonmissing elements of the queue
                * Height               - returns height of the queue
                */
    , priority /* NUMERIC, reflects added element's priority in the queue
                * for O, Output, D, Dequeue, R, Return and B, Bottom, or 
                * H, Head, T, Tail it holds a priority level of value 
                * popped from a queue
                * for I, Input, E, Enqueue, Insert it holds a priority 
                * level of value to be pushed into a queue
                * for C ignored 
                * for CHARACTERqueue and Sum, Nonmiss, Cnt, Avg, Mean, Average, Height 
                * returns calculated summary value
                */
    , value %qsysfunc(compress(&type., $, k)) 
               /* NUMERIC/CHARACTER  
                * for O, Output, D, Dequeue, R, Return and B, Bottom 
                * or H, Head, T, Tail it holds a value popped from a queue
                * for I, Input, E, Enqueue, Insert it holds a value 
                * to be pushed into a queue
                * for C ignored
                * for NUMERIC queue and Sum, Nonmiss, Cnt, Avg, Mean, Average, Height 
                * returns calculated summary value
                * otherwise does not modify value
                */
    );
    outargs value, priority;
/**************/

    length position positionTMP priority 8 value &type.;
    static position 0;
    declare hash H(ordered:"D", duplicate:"R", hashexp:&hashexp.);
    _RC_ = H.defineKey("priority");
    _RC_ = H.defineKey("position");
    _RC_ = H.defineKey("value");
    _RC_ = H.defineDone();
    declare hiter I("H"); 

    static _sum_ .;
    static _cnt_ .;
 
    select(upcase(IO));  
    /* Output from Top or Bottom - get the data from a queue */
      when ('O', 'OUTPUT', 'D', 'DEQUEUE', 'R', 'RETURN', 'B', 'BOTTOM')
      do;
        positionTMP = position;
        call missing(value, position);
        
        if IO =: 'B' or IO =: 'b' then
          do;
            _RC_ = I.last();
            _RC_ = I.next();
          end;
        else
          do;
            _RC_ = I.first();
            _RC_ = I.prev();
          end;

        %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
          %do; 
            /* since value is a character type then do nothing */
            _cnt_ = sum(_cnt_, (value ne " "));
          %end;
        %else /* numeric type */
          %do;
            _sum_ = sum(_sum_, -(value));
            _cnt_ = sum(_cnt_, -(value > .z));
          %end;        

        _RC_ = H.remove();
        position = positionTMP;

        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&queueName.] Debug O:" "dim(TEMP)=" _T_ "value=" value "position=" position;
        %end;
        return;
      end;
    
    /* Input - insert the data into a stack */
    when ('I', 'INPUT', 'E', 'ENQUEUE' ,'INSERT')
      do;      
        position = position + (&newOnTop.1);
        priority = coalesce(priority, 0);
        _RC_ = H.replace();

        %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
          %do; 
            /* since value is a character type then do nothing */
          %end;
        %else /* numeric type */
          %do; 
            _sum_ = sum(_sum_, (value));
            _cnt_ = sum(_cnt_, (value > .z));
          %end;

        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&queueName.] Debug I:" "dim(TEMP)=" _T_ "value=" value "position=" position;
        %end;
        return;
      end;

    /* Peek Top - peeks the data from a queue's top without removing */
    when ('H', 'HEAD')
      do;
        call missing(value, priority);
        _RC_ = I.first();
        _RC_ = I.prev();
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&queueName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
        %end;
        return;
      end;

    /* Peek Bottom - peeks the data from a queue's bottom without removing */
    when ('T', 'TAIL')
      do;
        call missing(value, priority);
        _RC_ = I.last();
        _RC_ = I.next();
        %if &debug %then %do;
          _T_ = H.num_items();
          put "NOTE:[&queueName.] Debug" "dim(TEMP)=" _T_ "TEMP[position]=" value;
        %end;
        return;
      end;

    /* Clear - reduce a stack to an empty one */
    when ('C', 'CLEAR')
      do;
        _RC_ = H.clear();
        position = 0;
        positionTMP = 0;
        _sum_ = .;
        _cnt_ = .;
        return;
      end;

    /* Statistic - returns selected statistic */
    %if %qsysfunc(compress(&type., $, k))=$ %then /* character type */
        %do; 
          when ('NONMISS', 'CNT')
            do;
              priority = _cnt_;
              return;
            end;
          when ('HEIGHT')
            do;
              priority = H.num_items();
              return;
            end;             
        %end;
      %else /* numeric type */
        %do; 
          when ('SUM') 
            do;
              value = _sum_; /* Sum */
              return;
            end;
          when ('AVG', 'MEAN', 'AVERAGE') 
            do;
              value = divide(_sum_, _cnt_); /* Average */
              return;
            end;         
          when ('NONMISS', 'CNT')
            do;
              value = _cnt_; /* NonMiss */
              return;
            end;
          when ('HEIGHT')
            do;
              value = H.num_items(); /* StackHeight */
              return;
            end;
        %end;

      otherwise;
    end;

    put "WARNING: IO parameter value" IO "is unknown.";
    put "NOTE: Use: 'O', 'OUTPUT', 'D', 'DEQUEUE', 'R', 'RETURN', 'B', 'BOTTOM'";
    put "NOTE:  or  'I', 'INPUT', 'E', 'ENQUEUE' ,'INSERT'";
    put "NOTE:  or  'T', 'TAIL', 'H', 'HEAD', 'C', 'CLEAR'";
    put "NOTE:  or  'SUM', 'AVG', 'MEAN', 'AVERAGE', 'NONMISS', 'CNT', 'HEIGHT'";
    return;
  endsub;

%if %superq(header) = 1 %then
%do;
  run;
%end;
%mend createDHPrtQueue;   

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Dynamic, Hash-based, and Character Priority queue:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHPrtQueue(PriorityQueuePC, type = $ 12, newOnTop=+);
  %createDHPrtQueue(PriorityQueueNC, type = $ 12, newOnTop=-); 
  options APPEND=(cmplib = WORK.DFAfcmp) ; 

  data Example1; 
   
    _I_ = .; 
    length _X_ _Y_ $ 3;
    do _X_ = "AAA","BBB","CCC","AA","BB","CC","A","B","C";
      _I_ + 1; 
      call PriorityQueuePC("I", mod(_I_, 3), _X_);
      call PriorityQueueNC("I", mod(_I_, 3), _X_); 
    end; 
   
    Height = .;
    call PriorityQueuePC('Height', Height, ''); 
    put Height=;

    do until(Height = 0); 
      call PriorityQueuePC('Dequeue', _I_, _X_);
      call PriorityQueueNC("Dequeue", _I_, _Y_);
      call PriorityQueueNC('Height', Height, '');
      put (_ALL_) (=); 
      output;  
    end; 
 
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Dynamic, Hash-based, and Numeric Priority queue:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDHPrtQueue(PriorityQueueN);
  options APPEND=(cmplib = WORK.DFAfcmp) ; 

  data Example2; 
   
    do _X_ = -5 to 5; 
      call PriorityQueueN("Enqueue", abs(_X_), _X_);
    end; 
   
    call missing(Sum, Avg, Cnt, Hgt); 
    call PriorityQueueN('Sum',    ., Sum);
    call PriorityQueueN('Avg',    ., Avg);
    call PriorityQueueN('Cnt',    ., Cnt);
    call PriorityQueueN('Height', ., Hgt);
    put (_ALL_) (=);

    do _N_ = 1 to Hgt; 
      call PriorityQueueN("Dequeue", _X_, _Y_);
      put _X_= _Y_=;
    end;  
  
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, July 2019.                            */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
