/*** HELP START ***//*
 
## >>> `%createDFBitmap()` macro: <<< <a name="createdfbitmap-macro"></a> #######################

The `%createDFBitmap()` macro allows to generate
a `dynamic function bitmap` which is a FCMP based 
approach to create *dynamically* allocated **numeric** 
bitmnap.

*Note:* Arrays provided by the macro are *one dimensional* arrays.

The idea of a SAS bitmap is based on:
1. SGF Paper 3101-2019
   titeled **Re-Mapping A Bitmap** by *Paul M. Dorfman* and *Lessia S. Shajenko* 
   [https://www.sas.com/content/dam/SAS/support/en/sas-global-forum-proceedings/2019/3101-2019.pdf](https://www.sas.com/content/dam/SAS/support/en/sas-global-forum-proceedings/2019/3101-2019.pdf)
2. SUGI Paper 8-26
   titeled **Table Look-Up by Direct Addressing: Key-Indexing -- Bitmapping -- Hashing** by *Paul M. Dorfman*
   [https://support.sas.com/resources/papers/proceedings/proceedings/sugi26/p008-26.pdf](https://support.sas.com/resources/papers/proceedings/proceedings/sugi26/p008-26.pdf)

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%createDFBitmap( 
  bitmapName           
 <,debug=0>
 <,outlib=work.DFAfcmp.package>
 <,type=32>
 <,header=1>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `bitmapName`   - *Required*, creates a FCMP call subroutine which is also 
                    a bitmap name. In the data step it is used in form of 
                    a call subroutine, e.g. `call bitmapName("Allocate", 3000)`.
                    Has to satisfy FCMP function naming requirements, but with 
                    maximum of 24 characters.

* `debug=`        - *Optional*, the default value is `0`.
                    If set to `1` then it turns on a debugging mode.

* `outlib=`       - *Optional*, the default value is `work.DFAfcmp.package`.
                    It points the default location for new generated dynamic 
                    function arrays compiled by FCMP.
                    *Hint!* Keep it as it is.

* `type=`         - *Optional*, the default value is `32`. Sets the type of 
                    bitwise operations executed internaly on the bitmap.
                    The only valid values are `32` or `52`, 
                    With 32 the `BOR()` and `BAND()` SAS functions are used
                    and with 52 the `bit64orDFA()` and `bit64and DFA()` FCMP 
                    functions are used.

* `header=`       - *Optional*, the default value is `1`. Indicates if 
                    the `proc fcmp outlib = &outlib.;` header is added to 
                    the executed code. If not 1 then no header is added.

**Created function arguments description**:

A function generated by the macro is:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
call &bitmapName.(IO, position, value)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
and accepts the following list of arguments and values:

1. `IO` - is a *character* steering argument, possible 
          values and behaviour they call are the following:
  - `Check`, `Test`, `T`     - to get information if a bit is set to 1 (on) or not,
  - `On`, `1`                - to set a selected bit to 1,
  - `Off`, `0`               - to set a selected bit to 0,
  - `C`, `Clear`             - to reduce a bitmat to a single empty cell,
  - `A`, `Allocate`          - to reserve space for a bitmap and set all bits to 0, 
  - `A0`, `Allocate0`        - to reserve space for a bitmap and set all bits to 0,
  - `A1`, `Allocate1`        - to reserve space for a bitmap and set all bits to 1,
  - `D`, `Dim`, `Dimension`  - to returns minimal and maximal index of the bitmap.
                       
2. `position` - is a *numeric* argument and depends on the `IO` value. 
                Behaves in the following way:
  - for `On`, `Off`, `1`, `0`/ `Check`, `Test`, `T` it is a bitmap index,
  - for `C`, `Clear` is ignored,
  - for `A`, `Allocate` sets the value of the minposition, i.e. the minimal position of the bitmap index,
  - for `D`, `Dimension` it returns value of the minposition,
            
.3 `value` - is a *numeric* argument and depends on the `IO` value. 
             Behaves in the following way:
  - for `Check`, `Test`, `T` it holds value retrieved from a bitmap on a given position,
  - for `On`, `Off`, `1`, `0`, `C`, `Clear` is ignored,
  - for `A`, `Allocate` it sets the value of the maxposition, i.e. maximal position of the array index,
  - for `D`, `Dimension` it returns value of the maxposition

The `position` and the `value` arguments are **outargs**, i.e. can be changed by the function.

*//*** HELP END ***/

%macro createDFBitmap(  /* macro Create Dynamic Function Array */
  bitmapName            /* array name, in datastep used in form of call subroutine
                          e.g. call arrayName("Allocate", 3000) 
                        */
, debug=0              /* if 1 then turns on debugging mode */

, outlib = work.DFAfcmp.package  /* default location for compiled functions */
, type=32                        /* default type of bitwise operations
                                    the only valid values are 32 or 52
                                    with 32: BOR() and BAND() functions are used
                                    with 52: bit64orDFA() and bit64and DFA() functions are used
                                  */
, header=1                       /* adding Proc FCMP header to the executed code 
                                    if other than 1 then _no_ "proc fcmp outlib = &outlib.;"
                                    will be added. 
                                  */
) / MINOPERATOR;

%if %superq(debug) NE 1 %then %let debug=0;

%if NOT (%superq(type) in (32 52)) %then %let type = 32;

%local BAND BOR;

%if &type. = 32 %then 
  %do;
    %let BAND = BAND;
    %let BOR  = BOR;
  %end;

%if &type. = 52 %then 
  %do;
    %let BAND = bit64andDFA;
    %let BOR  = bit64orDFA;
  %end;

/**************/
%if %superq(header) = 1 %then
%do;
  proc fcmp outlib = &outlib.;
%end;
  subroutine &bitmapName.(
      IO $     /* steering argument:
                * Check, Test, T       - get information if bit is set to 1 (on)
                * On, 1                - set selected bit to 1
                * Off, 0               - set selected bit to 0
                * C, Clear             - reduce a bitmat to a single empty cell
                * A, Allocate          - reserve space for bitmap and set all bits to 0 
                * A0, Allocate0        - reserve space for bitmap and set all bits to 0
                * A1, Allocate1        - reserve space for bitmap and set all bits to 1
                * D, Dim, Dimension    - returns minimal and maximal index of bitmap
                */
    , position /* for On, Off, 1, 0/ Check, Test, T it is a bitmap index
                * for C, Clear ignored
                * for A, Allocate sets value of minposition (i.e. minimal position of the bitmap index that occurred)
                * for D, Dimension returns minposition
                */
    , value    /* for Check, Test, T it holds value retrieved from a bitmap on a given position
                * for On, 1, Off, 0, C, Clear ignored
                * for A, Allocate sets value of maxposition (i.e. maximal position of the array index than occurred)
                * for D, Dimension returns maxposition
                */
    );
    outargs position, value;

/**************/

    array BM[1] / nosymbols; /* default size */
    static BM .;

    array BITMASK[&type.] / nosymbols; /* default size */
    static BITMASK .;
    array COBITMASK[&type.] / nosymbols; /* default size */
    static COBITMASK .;

    static maxposition 1; /* keep track of maximal position of the array index occurred */
    static minposition 1; /* keep track of minimal position of the array index occurred */
    static offset 0;      /* if array lower bound is less than 1 keep value of shift */


    X = ceil ((position + offset) / &type.) ;
    P = mod  ( position + offset  , &type.)+1 ;
    

    %if &debug %then %do;
      _T_ = dim(BM);
      put "NOTE:[&bitmapName.] Debug X i P:" "X=" X "P=" P;
    %end;

    select(upcase(IO));
    /* Output - get the data from an array 
     */
      when ('T', 'CHECK', 'TEST')
        do;
          if (minposition <= position <= maxposition) 
            then value = (&BAND.(BM[X], BITMASK[P]) > 0);
            else value = .;

          %if &debug %then %do;
            _T_ = dim(BM);
            put "NOTE:[&bitmapName.] Debug CHECK:" "dim(BM)=" _T_ "BM[X]=" BM[X] binary&type..;
            put "NOTE:[&bitmapName.] Debug CHECK:" "position=" position "value=" value;
          %end;
          return;
        end;

    /* ON - turn the bit on 
     */
      when ('ON', "1")
        do;
          if not(minposition <= position <= maxposition) then 
            do;
              put "ERROR: out of range!";
              put "ERROR: position should be between " minposition " and " maxposition;
              return;
            end;

          BM[X] = &BOR. (BM[x], bitmask[P]) ;


          %if &debug %then %do;
            _T_ = dim(BM);
            put "NOTE:[&bitmapName.] Debug ON: min=" minposition "and max=" maxposition;
            put "NOTE:[&bitmapName.] Debug ON: dim(BM)=" _T_ "value=" value "position=" position "BM[X]=" BM[X] binary&type..;
          %end;
          return;
        end;

    /* OFF - turn the bit off 
     */
      when ('OFF', "0")
        do;
          if not(minposition <= position <= maxposition) then 
            do;
              put "ERROR: out of range!";
              put "ERROR: position should be between " minposition " and " maxposition;
              return;
            end;

          BM[X] = &BAND. (BM[X], cobitmask[P]) ;

          %if &debug %then %do;
            _T_ = dim(BM);
            put "NOTE:[&bitmapName.] Debug OFF: min=" minposition "and max=" maxposition;
            put "NOTE:[&bitmapName.] Debug OFF: dim(BM)=" _T_ "value=" value "position=" position "BM[X]=" BM[X] binary&type..;
          %end;
          return;
        end;


    /* Allocate - reserve space for bitmap 
     *            and set starting value
     */
      when ('A', 'A1', 'A0', 'ALLOCATE', 'ALLOCATE1', 'ALLOCATE0' )
        do;
          if BITMASK[1] = . then
            do;
              %if &debug %then %do;
                put "NOTE: populating bitmask and cobitmask:";
              %end;
              do P = 1 to &type. ;
                BITMASK[P] = 2**(P-1) ;
                %if &debug %then %do;
                  put "NOTE-" bitmask[P] best32.;
                  put "NOTE-" bitmask[P] binary52.;
                %end;
                COBITMASK[P] = 4503599627370495 - 2**(P-1) ;
                %if &debug %then %do;
                  put "NOTE-" cobitmask[P] best32.;
                  put "NOTE-" cobitmask[P] binary52.;
                %end;
              end ;
            end;

          if .z < position <= value then 
            do;
              /* to handle the 65535 issue in older versions of SAS (<9.4M6) */
              _RESIZE_ = ceil(abs(value - position + 1)/&type.);
              if _RESIZE_ = 65535 then 
                do;
                  _RESIZE_ = _RESIZE_ + 1;
                  put "NOTE: to handle 65535 issue array size set to 65536";
                end;

              call dynamic_array(BM, _RESIZE_);
              if upcase(IO) in ('A1', 'ALLOCATE1') 
                then call fillmatrix(BM, 4503599627370495); /* to get 32/52 ones  */ 
                else call fillmatrix(BM, 0);                /* for 32/52 zeros */

              maxposition       = value;
              minposition       = position;
              offset            = 1 - position;

              %if &debug %then %do;
                _T_ = dim(BM);
                put "NOTE:[&bitmapName.] Debug A:" "dim(BM)=" _T_;
                put "NOTE:[&bitmapName.] Debug A:" "maxposition=" maxposition;
                put "NOTE:[&bitmapName.] Debug A:" "minposition=" minposition;
                put "NOTE:[&bitmapName.] Debug A:" "offset=" offset;
              %end;
              return;
            end;
          else 
            do;
              put "WARNING:" "Bitmap's lower bound must be less or equal than upper bound.";
              put "        " "Current values are: lower =" position " upper =" value;
              put "        " "&type. bit bitmap created.";
              call dynamic_array(BM, 1);
              maxposition       = &type.;
              minposition       = 1;
              BM[1]             = 0;
              offset            = 0;
              return;
            end;
        end;

    /* Clear - reduce an array to a single empty cell 
     */
      when ('C', 'CLEAR')
        do;
          call dynamic_array(BM, 1);
          maxposition       = &type.;
          minposition       = 1;
          BM[1]             = 0;
          offset            = 0;
          return;
        end;

    /* Dimension - returns minimal and maximal index 
     */
      when ('D', 'DIM', 'DIMENSION', 'DIMENSIONS')
        do;
          position = minposition;
          value    = maxposition;
          %if &debug %then %do;
            _T_ = dim(BM);
            put "NOTE:[&bitmapName.] Debug D:" "dim(BM)=" _T_;
          %end;
          return;
        end;
    end;


    put "WARNING: IO parameter value" IO "is unknown.";
    put "NOTE: Use: 'ON', '1', 'OFF', '0', 'T', 'CHECK', 'TEST'";
    put "NOTE:  or  'C', 'CLEAR', 'D', 'DIM', 'DIMENSION', 'DIMENSIONS'";
    put "NOTE:  or  'A', 'A0', 'A1', 'ALLOCATE', 'ALLOCATE0', 'ALLOCATE1'";
    return;
  endsub;

%if %superq(header) = 1 %then
%do;
  run;
%end;
%mend createDFBitmap;



/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Bitmap of type 32:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDFBitmap(MyBitmap32,type=32,debug=1);
  options APPEND=(cmplib = WORK.DFAfcmp) ;

  data Example1;
    call MyBitmap32("Allocate", -10, 100);
    L = 0; H = 0;
    call MyBitmap32("Dim", L, H);
    put L= H=;

 * populate array with data ;
    do i = L to H by 2;
      put i @;
      call MyBitmap32("1", i, .); 
    end;
    put;

 * get values from the array ;
    Value = .;
    do i = L to H;
      call MyBitmap32("T", i, Value); 
      put i= Value=;
    end;
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**EXAMPLE 2.** Bitmap of type 52:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDFBitmap(MyBitmap52,type=52,debug=1);
  options APPEND=(cmplib = WORK.DFAfcmp) ;

  data Example2;
    call MyBitmap52("Allocate", -10, 100);
    L = 0; H = 0;
    call MyBitmap52("Dim", L, H);
    put L= H=;

 * populate array with data ;
    do i = L to H by 2;
      put i @;
      call MyBitmap52("1", i, .); 
    end;
    put;

 * get values from the array ;
    Value = .;
    do i = L to H;
      call MyBitmap52("T", i, Value); 
      put i= Value=;
    end;
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**EXAMPLE 3.** Execution time test for type 52:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDFBitmap(MyBigBitmap52,type=52,debug=0);
  options FULLSTIMER APPEND=(cmplib = WORK.DFAfcmp) ;

  data Example3;
    call MyBigBitmap52("Allocate", -10, 2000000001);
    L = 0; H = 0;
    call MyBigBitmap52("Dim", L, H);
    put L= H=;

 * populate bitmap with data ;
    t = time();
    do i = L to H by 17;
      call MyBigBitmap52("1", i, .);
      x + 1; 
    end;
    t = time() - t;
    put "populate:" t= x=;

 * get values from the bitmap ;
    t = time();
    Value = .;
    do i = L to H;
      call MyBigBitmap52("T", i, Value); 
      x + (-Value);
    end;
    t = time() - t;
    put "search:" t= x=;
  run;

%*
L=-10 H=2000000001
populate:t=55.902999878 x=117647060
search:t=654.12900019 x=0
NOTE: The data set WORK.EXAMPLE3 has 1 observations and 6 variables.
NOTE: DATA statement used (Total process time):
      real time           11:50.42
      user cpu time       11:46.40
      system cpu time     0.45 seconds
      memory              301791.12k
      OS Memory           326332.00k
;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**EXAMPLE 4.** Execution time test for type 32:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %createDFBitmap(MyBigBitmap32,type=32,debug=0);
  options FULLSTIMER APPEND=(cmplib = WORK.DFAfcmp) ;

  data Example4;
    call MyBigBitmap32("Allocate", -10, 2000000001);
    L = 0; H = 0;
    call MyBigBitmap32("Dim", L, H);
    put L= H=;

 * populate bitmap with data ;
    t = time();
    do i = L to H by 17;
      call MyBigBitmap32("1", i, .);
      x + 1; 
    end;
    t = time() - t;
    put "populate:" t= x=;

 * get values from the bitmap ;
    t = time();
    Value = .;
    do i = L to H;
      call MyBigBitmap32("T", i, Value); 
      x + (-Value);
    end;
    t = time() - t;
    put "populate:" t= x=;
  run;

%*
L=-10 H=2000000001
populate:t=50.417999983 x=117647060
populate:t=611.13600016 x=0
NOTE: The data set WORK.EXAMPLE4 has 1 observations and 6 variables.
NOTE: DATA statement used (Total process time):
      real time           11:02.07
      user cpu time       10:59.07
      system cpu time     1.46 seconds
      memory              489583.90k
      OS Memory           513876.00k
;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since July 2019.                      */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
