/*** HELP START ***//*
 
## >>> `%GSM()` macro: <<< <a name="gsm-macro"></a> #######################

The `%GSM()` macro is the main macro of 
the **GSM** (a.k.a. *Generate Secure Macros*) package. 

It converts a list of macros provided by the user into
a data set of the Proc FCMP functions. The macros are stored
as encrypted code which allow to share the macros
without showing their code.

As a result a zip file, containing dataset with functions and 
code to be executed on site, is generated. 

Since encrypted code is stored in a SAS dataset it has
no limitation in sharing between operating systems (like catalogs have).

*Limitation:* Due to the `Resolve()` function limitations 
 a single macro file cannot be longer than 32760 bytes.

*Notes:* 
- All macros have to have the `secure` option added, i.e. `%macro aMacroname(...) / SECURE ;`.
- During the execution a test macro, named `%GSMpck_dummyMacroForTests()`, is generated.
- The `%GSM()` macro calls the `%GSMpck_makeFCMPcode(...)` macro internally.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%GSM( 
   path
 <,trim=0>
 <,cmplib=work.generateMacros>
 <,source2=>
 <,outpath=>
 <,encodingRestricted=>
 <,secret=>
 <,lineEnd=>
 <,encrypt=>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `path`               - *Required*, indicates a directory which contains files with macros.
                          Only files with `sas` extension are used.

* `cmplib=`             - *Optional*, the default value is `work.generateMacros`. 
                          Names the dataset which will contain generated functions. 

* `source2=`            - *Optional*, the default value is null. 
                          Indicate if `%includ`-ed files are printed out.
                          Any value other than null enables printing.

* `outpath=`            - *Optional*, the default value is set the same as the `path`.
                          Points a directory in which a result (a zip file) is generated.

* `encodingRestricted=` - *Optional*, the default value is `0`.
                          If set to 1 then if User session encoding is different from 
                          encoding of the session which generates the dataset then 
                          the generateMacros() function will not execute macro code.

* `secret=`             - *Optional*, the default value is null, in such case the 
                          secret is generated from the `sha256(datetime(), hex32.)` function 
                          and is printed in the log. When not null then should be
                          alphanumerical constant. Non-alphanumerical characters are removed.
                          Required to execute the `resolve()` function. 
                          User who do not know the value will not be able
                          to run the `_maxro_XX_()` function.

* `lineEnd=`            - *Optional*, the default value is `0D0A`, indicates which of:
                          line feed, carriage return, or both, or a space be inserted 
                          at the end of line in the intermediate code file that is generated.
                          Value has to be hexadecimal code (_NOT_ null),
                          since the value is resolved as `"&lineEnd."x`, so use e.g.
                          `0A` for line feed, `0D` for carriage return, 
                          `0D0A` for both, and `20` for space.

* `encrypt=`            - *Optional*, the default value is `ENCRYPT`. 
                          Indicate if `FCMP` functions generated by the package 
                          are encrypted. Value has to be either empty or `ENCRYPT`,
                          all other are converted to default. The option is 
                          dedicated for debugging, keep the default value 
                          for production use.

* `trim=`               - *Deprecated*, the default value is `0`.
                          *Kept for backward compatibility.*


---

*//*** HELP END ***/

%macro GSM( /* gsm.sas */
  path
, trim=0 /*0,1,2*/
, cmplib=work.generateMacros
, source2=
, outpath=
, encodingRestricted=0 /*0,1*/
, secret=
, lineEnd=0D0A
, encrypt=ENCRYPT
) / minoperator;
/* Parameters check */
%if %superq(path) = %then
  %do;
    %put NOTE: Use the helpPackage macro to get help info,;
    %put NOTE- e.g. %nrstr(%%helpPackage(GSM,%'%%GSM()%'));
    %goto exitPath;
  %end;
%else %if not %sysfunc(fileexist(&path.)) %then 
  %do;
    %put ERROR: [&sysmacroname.] The PATH &path does not exist. Aborting!;
    %goto exitPath;     
  %end;
%else %put NOTE: [&sysmacroname.] The PATH is &path;

%if %superq(encrypt) ne %then %let encrypt = ENCRYPT;

%if %superq(source2) ne %then %let source2 = source2;

%if (%superq(outpath) = ) %then 
  %do;
    %let outpath = &path.;
    %put NOTE: The OUTPATH set to:;
    %put NOTE- &path.;
  %end;
%else %if not %sysfunc(fileexist(&outpath.)) %then 
  %do;
    %let outpath = &path.;
    %put WARNING: The OUTPATH does not exist, setting to:;
    %put WARNING- &path.;
  %end;


%if not (%superq(trim) in (0 1 2)) %then
  %do;
    %let trim = 0;
    %put NOTE: [&sysmacroname.] The TRIM= parameter is deprecated.;
  %end;
%let trim = 0;

%if not (%superq(encodingRestricted) in (0 1)) %then
  %do;
    %let encodingRestricted = 0;
    %put WARNING: [&sysmacroname.] The encodingRestricted= parameter set to default (i.e. 0).;
  %end;

%if (%superq(encodingRestricted) = 1) %then
  %do;
    %put NOTE: [&sysmacroname.] The encodingRestricted= parameter is set to 1.;
    %put NOTE- [&sysmacroname.] This will prevent the generateMacros() function from executing;
    %put NOTE- [&sysmacroname.] if User session encoding is different than current session encoding.;
    %put NOTE- [&sysmacroname.] Current session encoding is &sysencoding..;
  %end;

%if %superq(secret) = %then
  %do;
    data _null_;
      call symputX("secret"
                   ,put(sha256(cats(datetime())), hex32.)
                   ,"L");
    run;
  %end;
%else
  %do;
    data _null_;
      call symputX("secret"
                   ,upcase(compress(symget("secret"), ,"KAD"))
                   ,"L");
    run;
  %end;

%put NOTE-;
%put NOTE: [&sysmacroname.] Value of secret is: &secret.;
%put NOTE-;

/* the code */
%local filesWithCodes fileNameCode outlib tmp_syslst tmp_outlib tmp_notes;
%let tmp_notes = %sysfunc(getoption(notes));
options noNotes;

data _null_;
/* test cmplib dataset name */
  test = "&cmplib.";
  a = lengthn(quote(compress(strip(test),".","KFN"),"'")) ^= lengthn(quote(strip(test),"'"));
  b = find(test,".",1)>9;
  c = not(lengthn(test)<42);
  i = countw(test,".","m");
  d = i>2;

  e = 0;
  do i = 1 to i;
    e + input(char(scan(test,i,".","m"),1), ?? best.) in (0 1 2 3 4 5 6 7 8 9);
    c + not(0<lengthn(scan(test,i,".","m"))<33);
  end;

  x = a + b + c + d + e;
  /*put (_ALL_) (=);*/
  if x > 0 then 
    do;
      put "ERROR: The " test "is invalid SAS dataset name!";
      call symputx("cmplib",'work.generateMacros',"L");
      put "ERROR- The default work.generateMacros will be used.";
    end;
run;

%let tmp_syslst = &syslast.;
%let syslast = &cmplib.;
%let cmplib  = &syslast.;
%let syslast = &tmp_syslst.;
%let outlib  = &cmplib..secure;

%let filesWithCodes = WORK._DATA_%sysfunc(datetime(), hex16.)_;
%let fileNameCode   = _%sysfunc(datetime(), hex6.)c;

data &filesWithCodes.;
  base = "&path.";
  putlog "*" base +(-1) "*";
  length file $ 256;

  folderRef = "_%sysfunc(datetime(), hex6.)0";

  rc = filename(folderRef, base);
  folderid = dopen(folderRef);

  do i=1 to dnum(folderId); drop i;
    file = dread(folderId, i);
    if upcase(scan(file, -1, ".")) = "SAS" then output;
  end;

  rc = filename(folderRef);
keep base file;
run;
proc sort data = &filesWithCodes.;
  by file;
run;
title "&filesWithCodes.";
proc print data = &filesWithCodes. width=full;
run;

filename &fileNameCode. TEMP;


data _null_;
  file &fileNameCode.;
  put "/* Code to be run locally. */";
  put " ";
run;

data _null_;
 set &filesWithCodes.;
 call execute(cats(
    '%nrstr(%GSMpck_makeFCMPcode(', base, '/', file, ',', _N_
  , ", trim=&trim., outlib=&outlib., source2=&source2., fileNameCode=&fileNameCode., secret=&secret., lineEnd=&lineEnd.))"
  ));
run;

%let tmp_cmplib = %sysfunc(getoption(cmplib, keyword));
options cmplib = _null_;
proc FCMP outlib = &outlib. inlib = &cmplib. &encrypt. ;
  function generateMacros();
    if symget('sysencoding') ne "&sysencoding." then
      do;
        %if (%superq(encodingRestricted) = 1) %then
          %do;
            put "ERROR: Current system encoding is different than &sysencoding.!";
            put "ERROR- Macros will not be generated.";
            return(-1);
          %end;
        %else
          %do;
            put "WARNING: Current system encoding is different than &sysencoding..";
            put 'NOTE:    It may affect characters that are "encoding specific".';
            put "NOTE:    The best solution is to run SAS session in &sysencoding. encoding.";
          %end;
      end; 
 
    %include &fileNameCode. / &source2.;
    return(0);
  endsub;
run;
data _null_; run;
%local tmp_err; /* check if there were any errors during macro generation */
options cmplib = &cmplib.;
data _null_;                 
  rc = generateMacros();     
run;                         
%let tmp_err=&syserr.;     
data _null_; run;            

options &tmp_cmplib.;

%if not (0 < &tmp_err.) %then
  %do;
    %local codeZIP cmplibDS;
    %let codeZIP  = _%sysfunc(datetime(), hex6.)z;
    %let cmplibDS = %lowcase(%scan(&cmplib., -1, "."));

    filename &codeZIP. ZIP "&outpath./&cmplibDS..zip";

    data _null_;
      /* copy the files with data */
      do ext = "sas7bndx", "sas7bdat";
        in  = "_%sysfunc(datetime(), hex6.)i";
        out = "_%sysfunc(datetime(), hex6.)o";

        call execute("filename " 
                  !! in  
                  !! ' "' 
                  !! pathname(scan("&cmplib.", 1, ".")) !! "/&cmplibDS.." 
                  !! strip(ext) 
                  !! '" lrecl=1 recfm=n;');
        call execute("filename " 
                  !! out 
                  !! " ZIP '&outpath./&cmplibDS..zip' " 
                  !! "member='&cmplibDS.." 
                  !! strip(ext) 
                  !! "' lrecl=1 recfm=n;");

        call execute('data _null_;');
        call execute(' rc1 = fexist("' !! in  !! '");');
        call execute(' rc2 = fcopy("' !! in !! '", "' !! out !! '");');
        call execute(' rc3 = fexist("' !! out !! '");');
        call execute(' putlog (rc:) (=);');
        call execute('run;');

        call execute("filename " !! in  !! " clear;");
        call execute("filename " !! out !! " clear;");
      end;

      /* prepare sas file with code */
      file &codeZIP.(&cmplibDS..sas);
      do text =
        "/*-------------------------------------------------------------------------*/",
        "/* Code to be run on site. Remember to change the Path for the proper one. */",
        "/*-------------------------------------------------------------------------*/",
        "                                                       ",
        "libname code '<...path...>' inencoding='&sysencoding.';",
        "proc sort                        ",
        " data = code.&cmplibDS.          ",
        " out = code.&cmplibDS.(          ",
        " type=etsmodel                   ",
        " compress=char                   ",
        " pointobs=yes                    ",
        " index=(_Key_)                   ",
        " encoding='&sysencoding.')       ",
        " force;                          ",
        "  by _Key_ Sequence;             ",
        "run;                             ",
        "options cmplib = code.&cmplibDS.;",
        "data _null_;                     ",
        "  rc = generateMacros();         ",
        "run;                             ",
        "data _null_; run;                ",
        "                                 ";
        putlog "WARNING-" text $char100.;
        put text $char100.;
      end;
      stop;
    run;
  %end;
%else %put WARNING: The generation process was not clean. ZIP file was not created.;

proc delete data = &filesWithCodes.;
run;

options &tmp_notes.;
%macro GSMpck_dummyMacroForTests; %mend;
data _null_; run;

%exitPath:
%mend GSM;


/*** HELP START ***//*

### Example: ###################################################################

Example 1. Prepare 2 files: `f1.sas` and `f2.sas` and use the `%GSM()` macro.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%let path = %sysfunc(pathname(work))/path2files;

%put &=path.;
options dlcreatedir;
libname path "&path.";
filename path "&path.";

data _null_;
  file path(f1.sas);
  input;
  put _infile_;
cards4;
%macro abc(x) / SECURE;
  data test;
    do i = 1 to &x.;
      put i=;
    end;
  run;
%mend;
;;;;
run;

data _null_;
  file path(f2.sas);
  input;
  put _infile_;
cards4;
%macro xyz(x) / SECURE;
  %do i = 1 %to &x.;
    %put &=i;
  %end;
%mend;
;;;;
run;

%GSM(&path., cmplib=work.myMacros)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

*//*** HELP END ***/
