/*** HELP START ***//*
 
## >>> `%appendCell()` macro: <<< <a name="appendcell-macro"></a> ##############

The `%appendCell()` macro allows to append
a macrovariable to a macroarray created by the `%array()` macro.

Dimensions of the macroarray are extended.

The `%appendCell()` macro executes like a pure macro code.

### SYNTAX: ####################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%appendCell(
  first
 ,second
 ,hilo
) 
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `first` -  *Required*, a name of a macroarray created by the `%array()` macro.

2. `second` - *Required*, a name of a macrovariable to be append to the macroarray.

3. `hilo` -   *Required*, if `H` macrovariable is appended at the end 
                          if `L` macrovariable is appended at the beginning 

*//*** HELP END ***/

%macro appendCell(
  first
, second
, hilo
);
%local rc;
%if (NOT %sysevalf(%superq(first)=)) and (NOT %sysevalf(%superq(second)=)) %then
  %do;
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    %if %SYMEXIST(&first.LBOUND)  and %SYMEXIST(&first.HBOUND)  and %SYMEXIST(&first.N)
    and %SYMEXIST(&second.)
    %then
      %do;
        %let rc = %sysfunc(dosubl(
        /*===============================================================================================*/
          options nonotes nosource ps=min ls=max %str(;)
          data _null_ %str(;)
          /**/
            select %str(;)
              when (UPCASE("&hilo.") = "H")
                do %str(;)
                  call symputX( cats("&first.", put(&&&first.HBOUND + 1, best32.))
                          , symget( "&second." )
                          , "G") %str(;)
                  call symputX("&first.HBOUND", put(&&&first.HBOUND + 1, best32.), "G") %str(;)
                  call symputX("&first.N", put(&&&first.N + 1, best32.), "G") %str(;)
                end %str(;)
              when (UPCASE("&hilo.") = "L")
                do %str(;)
                  if &&&first.LBOUND > 0 then
                    do %str(;)
                      call symputX( cats("&first.", put(&&&first.LBOUND - 1, best32.))
                            , symget( "&second." )
                            , "G") %str(;)   
                      call symputX("&first.LBOUND", put(&&&first.LBOUND - 1, best32.), "G") %str(;)
                      call symputX("&first.N", put(&&&first.N + 1, best32.), "G") %str(;)  
                    end %str(;)
                  else
                    do %str(;) 
                      putlog "*" %str(;)
                      putlog "* EFFICIENCY INFORMATION NOTE:[&SYSMACRONAME.] LBOUND is less than 0. Arrays will be aligned right." %str(;)
                      putlog "*" %str(;)
                      do i = (&&&first.HBOUND) to (&&&first.LBOUND) by -1 %str(;)
                        call symputX( cats("&first.", put(i+1, best32.))
                                    , symget( cats("&first.", put(i, best32.)) )
                                    , "G") %str(;)
                      end %str(;)
                      call symputX("&first.0", symget( "&second." ), "G") %str(;)
                      call symputX("&first.HBOUND", put(&&&first.HBOUND + 1, best32.), "G") %str(;)
                      call symputX("&first.N", put(&&&first.N + 1, best32.), "G") %str(;)
                    end %str(;)
                end %str(;)
              otherwise 
                do %str(;)  
                  putlog "ERROR:[&SYSMACRONAME.] The HILO argument must be H or L." %str(;) 
                  putlog "ERROR-[&SYSMACRONAME.] No modification done." %str(;)
                end %str(;)  
            end %str(;)
          run %str(;)
        /*===============================================================================================*/
        ));
      %end;
    %else
      %do;
        %put ERROR:[&SYSMACRONAME.] The First or the Second array is undefined.;
        %put ERROR-[&SYSMACRONAME.] No modification done.;
      %end;
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/  
  %end;
%else
  %do;
    %put WARNING:[&SYSMACRONAME.] The First or the Second argument is missing.;
    %put WARNING-[&SYSMACRONAME.] No modification done.;
  %end;
%mend appendCell;


/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Create two macro wrappers.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %* Macro wrapper to append a macrovariable to the end of a macroarray;
  %macro appendHC(array,cell);
    %appendCell(&array.,&cell.,H)
  %mend appendHC;

  %* macro wrapper to append a macrovariable to the beginning of a macroarray;
  %macro appendLC(array,cell);
    %appendCell(&array.,&cell.,L)
  %mend appendLC;


  %* create macroarrays X and variables W,Y,Z;

  %array(X[2:4] $ ("AAA", "BBB", "CCC"), macarray=Y)
  %let W=1;
  %let Y=2;
  %let Z=3;
  %put *%do_over(X)*&=W*&=Y*&=Z*;

  %put BEFORE *%do_over(X)**&=xLBOUND*&=xHBOUND*&=xN*;
  %appendCell(X,Y,H)
  %put AFTER1 *%do_over(X)**&=xLBOUND*&=xHBOUND*&=xN*;

  %appendLC(X,W)
  %put AFTER2 *%do_over(X)**&=xLBOUND*&=xHBOUND*&=xN*;

  %appendHC(X,Z)
  %put AFTER3 *%do_over(X)**&=xLBOUND*&=xHBOUND*&=xN*;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Error handling
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %appendCell(X,Y,blahblah)

  %appendCell(X,,H)
  %appendCell(,Y,H)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**EXAMPLE 3.** Adding variable below lower bound.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(zero[0:2] $ ("AAA", "BBB", "CCC"), macarray=Y)
  %let belowzero=zzz;

  %put BEFORE *%do_over(zero)**&=zeroLBOUND*&=zeroHBOUND*&=zeroN*;
  %appendCell(zero,belowzero,L)
  %put AFTER  *%do_over(zero)**&=zeroLBOUND*&=zeroHBOUND*&=zeroN*;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, February 2019.                        */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
