/*** HELP START ***//*
 
## >>> `%array()` macro: <<< <a name="array-macro"></a> #######################

The code of a macro was inspired by 
*Ted Clay's* and *David Katz's* macro `%array()`. 

The `%array()` macro version provided in the package 
is designed to facilitate 
the idea of macroarray concept, i.e. *a list of 
macrovariables with common prefix and numerical suffixes*.
Usually such construction is then resolved by 
double ampersand syntax, e.g. `&&perfix&i` or similar one.

What is new/extension to the `%array()` macro concept are:

0. The syntax is closer to the data step one.
1. It is a pure macro code (it can be executed in any place 
   of 4GL code), this includes generating macroarrays out 
   of datasets. 
2. When a macroarrray is created it allows also to generate
   a new macro (named the same as the array name) and replace 
   the double ampersand syntax with more array looking one,
   i.e. for array ABC user can have `%ABC(1)`, `%ABC(2)`, or `%ABC(&i)`
   constructions.
3. The array macro allows to use data step functions to generate 
   array's entries.

The `%array()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%array(
   array
  <,function=>
  <,before=>
  <,after=>
  <,vnames=N>
  <,macarray=N>
  <,ds=>
  <,vars=>
  <,q=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `array` -      *Required*, an array name and a declaration/definition of an array, <br>
                  e.g. `myArr[*] x1-x3 (4:6)` <br>
                    or `myBrr[*] $ y1-y3 ("a" "b" "c")` <br>
                    or `myCrr[3] $ ("d d d" "e,e,e" "f;f;f")` <br>
                    or `myDrr p q r s`. <br>
                  Macrovariables created by the macro are *global*.
                  If an array name is `_` (single underscore) then attached variables 
                  list names are used, a call of the form: 
                    `%array(_[*] p1 q2 r3 s4 (-1 -2 -3 -4))` 
                  will create macrovariables: `p1`, `q2`, `r3`, and `s4` with respective 
                  values: `-1`, `-2`, `-3`, and `-4`. <br>
                  Three additional *global* macrovariables: 
                    `<arrayName>LBOUND`, `<arrayName>HBOUND`, and `<arrayName>N` 
                  are generated with the macroarray. See examples for more use-cases.

* `function=` -   *Optional*, a function or an expression to be applied to all array cells, 
                  `_I_` is as array iterator, e.g. `_I_ + rand("uniform")`.

* `before=` -     *Optional*, a function or an expression to be added before looping through 
                  array, e.g. `call streaminit(123)`.

* `after=` -      *Optional*, a function or an expression to be added after looping through 
                  array, e.g. `call sortn(ABC)`.

* `vnames=N` -    *Optional*, default value `N`, if set to `Y`/`YES` then macroarray is built based 
                  on variables names instead values, e.g. 
                    `%array(myArr[*] x1-x3 (4:6), vnames=Y)`
                  will use `x1`, `x2`, and `x3` as values instead `4`, `5`, and `6`.

* `macarray=N` -  *Optional*, default value `N`, if set to `Y`/`YES` then a macro, named with the array 
                  name, is compiled to create convenient envelope for multiple ampersands, e.g. 
                    `%array(myArr[*] x1-x3 (4:6), macarray=Y)` 
                  will create `%myArr(J)` macro which will allow to extract "data" 
                  from macroarray like: 
                    `%let x = %myArr(1);` 
                  or when used with second parameter equal `I` (insert) allow to overwrite macroarrays 
                  value: 
                    `%let %myArr(17,i) = 42;`
                  If set to `M` then for a given array name the macro symbols table is scanned for
                  macrovariables with prefix like the array name and numeric suffixes, 
                  then the minimum and the maximum index is determined
                  and all not existing global macrovariables are created and
                  a macro is generated in the same way as for the `Y` value.

* `ds=` -        *Optional*, use a dataset as a basis for a macroarray data, 
                 if used by default overwrites use of the `array` parameter, honors `macarray=` 
                 argument, dataset options are allowed, e.g. `sashelp.class(obs=5)`

* `vars=` -      *Optional*, a list of variables used to create macroarrays from a dataset, 
                 the list format can be as follows (`<...>` means optional):
                   `variable1<delimiter><arrayname1> <... variableN<delimiter><arraynameN>>`
                 delimiters are hash(`#`) and pipe(`|`), currently only space 
                 is supported as separator, the meaning of `#` and `|` delimiters 
                 will be explained in the following example: 
                 if the `vars = height#h weight weight|w age|` value is provided
                 then the following macroarrays will be created: <br>
                  1) macroarray "H" with ALL(`#`) values of variable "height"  <br>
                  2) macroarray "WEIGHT" with ALL(no separator is equivalent to #) 
                     values of variable "weight" <br>
                  3) macroarray "W" with UNIQUE(|) values of variable "weight" and <br>
                  4) macroarray "AGE" with UNIQUE(|) values of variable "age".

* `q=` -        *Optional*, indicates (when set to `1` or '2') if the value should be surrounded by quotes.
                It uses `quote(cats(...))` combo under the hood. Default value is `0`.
                Value `1` is for apostrophes, value `2` is for double quotes.
                Ignored for `macarray=M`.

*//*** HELP END ***/

%macro array(
   array
  ,function=
  ,before=
  ,after=
  ,vnames=N
  ,macarray=N
  ,ds=
  ,vars=
  ,q=0
)
/
minoperator
; 
%local array function before macarray ds dsID vars
       name _BEFORE_ _F_ _FBY_ _AFTER_ rc mtext
       _ALL_MACROVARIABLES_CREATED_;

%if NOT (%superq(q) IN (1 2)) %then %let q=0;

%if %superq(ds)   ne           %then %goto FromDataset;
%if %superq(array) =           %then %goto Exit;
%if %superq(macarray) IN (m M) %then %goto MakeMacro;

/* CREATE AN ARRARY FROM A STATEMENT */
/* extract an array name */
%let name = %unquote( %qscan( &array., 1, %str([{( ;)}]) ) );
/* %put **&=array.**; */
/* %put **&=name.**; */

/* temp macrovariable */
%let mtext = _%sysfunc(int(%sysevalf(1000000 * %sysfunc(rand(uniform)))))_;

/* you can insert a function() into array */
%if %superq(function) ne %then
  %do;
    %let _F_ = &name.[_I_] = %superq(function) ;
    %let _FBY_ = 1;
  %end;
%else 
  %do;
    %let _F_ =;
    %let _FBY_ = -1;
  %end;

/* sometimes function needs e.g. call streaminit() */
%if %bquote(&before.) ne %then %let _BEFORE_ = %superq(before);
                         %else %let _BEFORE_ =;
/* sometimes array needs e.g. call sortn() */
%if %bquote(&after.) ne  %then %let _AFTER_ = %superq(after);
                         %else %let _AFTER_ =;
%let rc = %sysfunc(
dosubl(
/*===============================================================================================*/
options nonotes nosource ps=min%str(;)
DATA _NULL_ %str(;)
 ARRAY %superq(array) %str(;) /* here the array is created */

 /* if range starts < 0 the it is shifted to 0 */
 _IORC_ = IFN(LBOUND(&name.) < 0, -LBOUND(&name.), 0) %str(;)

 %unquote(&_BEFORE_.) %str(;)

 DO _I_ = LBOUND(&name.) TO HBOUND(&name.) BY &_FBY_. %str(;)
   %unquote(&_F_.) %str(;) /* place for a function to be applied */
 END %str(;)

 %unquote(&_AFTER_.) %str(;)



 if "&q." in ('1',"2") then
  do %str(;)
   /* add quotes by: quote(cats(...))*/
   DO _I_ = LBOUND(&name.) TO HBOUND(&name.) %str(;)
     SELECT %str(;)
        WHEN ("&name." = "_") /* if array name is _ then use variables names instead as names */
          CALL SYMPUTX(VNAME(&name.[_I_]), quote(cats(&name.[_I_])
                            ,ifc("&q."='1',"'",'"')), 'G') %str(;)
        WHEN ("%qupcase(&vnames.)" in ("Y" "YES")) /* use variables names as values */
          CALL SYMPUTX(CATS("&name.", put(_I_+_IORC_, best32.))
                           ,quote(cats(VNAME(&name.[_I_])), ifc("&q."='1',"'",'"')), 'G') %str(;)
        OTHERWISE
          CALL SYMPUTX(CATS("&name.", put(_I_+_IORC_, best32.))
                           ,quote(cats(&name.[_I_]), ifc("&q."='1',"'",'"')), 'G') %str(;)
     END %str(;)
   END %str(;)
  end %str(;)
 else
  do %str(;)
   DO _I_ = LBOUND(&name.) TO HBOUND(&name.) %str(;)
     SELECT %str(;)
        WHEN ("&name." = "_") /* if array name is _ then use variables names instead as names */
          CALL SYMPUTX(VNAME(&name.[_I_]), &name.[_I_], 'G') %str(;)
        WHEN ("%qupcase(&vnames.)" in ("Y" "YES")) /* use variables names as values */
          CALL SYMPUTX(CATS("&name.", put(_I_+_IORC_, best32.)), VNAME(&name.[_I_]), 'G') %str(;)
        OTHERWISE
          CALL SYMPUTX(CATS("&name.", put(_I_+_IORC_, best32.)), &name.[_I_], 'G') %str(;)
     END %str(;)
   END %str(;)
  end %str(;)

 CALL SYMPUTX("&name.LBOUND", LBOUND(&name.)+_IORC_, 'G') %str(;)
 CALL SYMPUTX("&name.HBOUND", HBOUND(&name.)+_IORC_, 'G') %str(;)
 CALL SYMPUTX("&name.N", HBOUND(&name.) - LBOUND(&name.) + 1, 'G') %str(;)
 CALL SYMPUTX ("_all_macrovariables_created_", HBOUND(&name.) - LBOUND(&name.) + 1, "L") %str(;)
 /* create a macro which allows to call to created macroarrayarray 
    elements like to a regular array e.g. '%let x = %arr(17);' */
 IF NOT("&name." = "_") AND ("%qupcase(&macarray.)" in ("Y" "YES"))
 THEN
   CALL SYMPUTX("&mtext.", 
        '%MACRO ' !! "&name." !! '(J,M);' !! 
        '%local J M; %if %qupcase(&M.)= %then %do;' !! /* empty value is output, I is input */
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&&&sysmacroname.&J.%end;' !! 
        '%else %do;' !! 
           /* put . for numeric "out of range" */
           IFC(not(UPCASE("&vnames.") = "Y") AND  VTYPE(&name.[LBOUND(&name.)])='N', '.', '') !!
          '%put WARNING:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Missing value is used.;' !!
        '%end;' !!
        '%end;' !!
        '%else %do; %if %qupcase(&M.)=I %then %do;' !!
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&sysmacroname.&J.%end;' !! 
        '%else %do;' !! 
          '%put ERROR:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put ERROR-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
        '%end;' !!
        '%end; %end;' !!
        '%MEND;', 'G') %str(;)
 ELSE
   CALL SYMPUTX("&mtext.", ' ', 'G') %str(;)
RUN %str(;)
/*===============================================================================================*/
));
&&&mtext.
%symdel &mtext. / NOWARN ;
%put NOTE:[&sysmacroname.] &_ALL_macrovariables_CREATED_. macrovariables created;
%goto Exit;


%MakeMacro: 
/* CREATE AN ARRAY FROM EXISTING LIST OF MACROVARIABLES */
/* extract an array name */
%let name = %unquote( %qscan( &array., 1, %str([{( )}]) ) );
/* %put **&=array.**; */
/* %put **&=name.**; */

/* temp macrovariable */
%let mtext = _%sysfunc(int(%sysevalf(1000000 * %sysfunc(rand(uniform)))))_;

%let rc = %sysfunc(
dosubl(
/*===============================================================================================*/
options nonotes nosource ps=min%str(;)
PROC SQL %str(;)
  create table work.&mtext.1 as
  select distinct name 
  from dictionary.macros
  where name like "%upcase(&name.)" !! '%'
    and scope = 'GLOBAL'
  %str(;)
QUIT %str(;)
DATA work.&mtext.1 %str(;)
  IF end THEN OUTPUT %str(;)
  SET work.&mtext.1 end = end %str(;)
  id = input(substrn(name, %eval(%length(&name.)+1)), ?? best32.) %str(;)
  IF id %str(;)
  retain min max %str(;)
  min = min(min, id) %str(;)
  max = max(max, id) %str(;)
  keep min max %str(;)
RUN %str(;) 
/*
PROC PRINT DATA = work.&mtext.1 %str(;)
RUN %str(;)
*/ 
DATA _NULL_ %str(;)
 SET work.&mtext.1 end = end %str(;)
 CALL SYMPUTX("&name.LBOUND", min, 'G') %str(;)
 CALL SYMPUTX("&name.HBOUND", max, 'G') %str(;)
 CALL SYMPUTX("&name.N", max - min + 1, 'G') %str(;)
 CALL SYMPUTX ("_all_macrovariables_created_", max - min + 1, "L") %str(;)
 /* create a macro which allows to call to created macroarrayarray 
    elements like to a regular array e.g. '%let x = %arr(17);' */
 IF NOT("&name." = "_")
 THEN
   CALL SYMPUTX("&mtext.", 
        '%MACRO ' !! "&name." !! '(J,M);' !! 
        '%local J M; %if %qupcase(&M.)= %then %do;' !! /* empty value is output, I is input */
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&&&sysmacroname.&J.%end;' !! 
        '%else %do;' !!  
          '%put WARNING:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Missing value is used.;' !!
        '%end;' !!
        '%end;' !!
        '%else %do; %if %qupcase(&M.)=I %then %do;' !!
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&sysmacroname.&J.%end;' !! 
        '%else %do;' !! 
          '%put ERROR:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put ERROR-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
        '%end;' !!
        '%end; %end;' !!
        '%MEND;', 'G') %str(;)
 ELSE
   CALL SYMPUTX("&mtext.", ' ', 'G') %str(;)
 STOP %str(;)
RUN %str(;)
PROC DELETE DATA = work.&mtext.1 %str(;)
RUN %str(;)
/*===============================================================================================*/
));
%do _F_ = &&&name.LBOUND %to &&&name.HBOUND;
  %global &name.&_F_.;
%end; 
&&&mtext.
%symdel &mtext. / NOWARN ;
%put NOTE:[&sysmacroname.] &_ALL_macrovariables_CREATED_. macrovariables created;
%goto Exit;

%FromDataset: 
/* CREATE AN ARRAY FROM A DATASET */

/* for example:                                               */
/* let ds = sashelp.class                                     */
/* if vars = height#h weight weight|w age|
   then create:
    1) marray "h" with ALL values of variable "height"
    2) marray "weight" with ALL values of variable "weight"
    3) marray "w" with UNIQUE values of variable "weight"
    4) marray "age" with UNIQUE values of variable "age"
*/

%local numvars i toBeCalled toBeLooped toBeAfter av avNumber a v;

/* count number of variables */
%let numvars = %qsysfunc(countw(%superq(vars), %str( )));
%if not &numvars. %then 
%do;
  %put NOTE:[&sysmacroname.] No variables listed! Exiting.;
  %goto Exit;
%end;
/*%put &=numvars.;*/

/* verify existence of &ds. */
%if NOT %sysfunc(exist(%scan(%superq(ds),1,()))) %then
%do;
  %put WARNING:[&sysmacroname.] Dataset %superq(ds) does NOT exist! Exiting.;
  %goto Exit;
%end;

/* verify the number of observations in &ds. */
%let dsID = %sysfunc(open(&ds.,I));
%if &dsID. %then
  %do;
    %if 1 > %sysfunc(ATTRN(&dsID., ANY)) %then
      %do;
        %put WARNING:[&sysmacroname.] Dataset %superq(ds) does not have observations or variables... Exiting.;
        %let dsID = %sysfunc(close(&dsID.));
        %goto Exit;
      %end;
      %let dsID = %sysfunc(close(&dsID.));
  %end;
%else
  %do;
    %put WARNING:[&sysmacroname.] Cannot open dataset %superq(ds)! Exiting.;
    %goto Exit;
  %end;

/********************************************************************************/


%let toBeCalled =; /* declarations before looping through dataset */
%let toBeLooped =; /* declarations during looping through dataset */
%let toBeAfter =;  /* declarations after looping through dataset */

/* loop through all variables */
%do i = 1 %to &numvars.;
  /* select variable and check if there is differnt name for array  */
  %let av = %qscan(%superq(vars), &i., %str( )); 
  %let avNumber = %qsysfunc(countw(%superq(av), %str(#|)));
  /*%put *&=av*&=avNumber*;*/
  
  /* get variable name and array name */
  %let v = %qscan(%superq(av),          1, %str(#|));
  %let a = %qscan(%superq(av), &avNumber., %str(#|));

  /* check if unique values should be selected */
  %if %index(%superq(av), %str(|)) > 0 %then
    %do; 
      /*%put *in pipe|*&=a &=v*;*/
      %let toBeCalled 
    = %superq(toBeCalled)%str(;) 
      declare hash &v._h&i.() %str(;) 
      &v._h&i..defineKey("&v.") %str(;) 
      &v._h&i..defineData("&v.") %str(;) 
      &v._h&i..defineDone() %str(;)
      declare hiter &v._hi&i.("&v._h&i.") %str(;) 
      ;
      %let toBeLooped 
    = %superq(toBeLooped)%str(;)
      _IORC_ = &v._h&i..ADD() %str(;)
      ;
      %let toBeAfter 
    = %superq(toBeAfter)%str(;)
      CALL SYMPUTX("&a.LBOUND",                  1, 'G') %str(;)
      CALL SYMPUTX("&a.HBOUND", &v._h&i..NUM_ITEMS, 'G') %str(;)
      CALL SYMPUTX("&a.N"     , &v._h&i..NUM_ITEMS, 'G') %str(;)

      /* quote(cats(...)) */
      if "&q." in ('1', "2") then
       do %str(;)
        DO WHILE(&v._hi&i..NEXT()=0) %str(;)
          &v._i&i.+1%str(;) CALL SYMPUTX(CATS("&a.", put(&v._i&i., best32.))
                                        ,quote(cats(&v.), ifc("&q."='1',"'",'"')), 'G') %str(;)
        END %str(;)
       end %str(;)
      else
       do %str(;)
        DO WHILE(&v._hi&i..NEXT()=0) %str(;)
          &v._i&i.+1%str(;) CALL SYMPUTX(CATS("&a.", put(&v._i&i., best32.)), &v., 'G') %str(;)
        END %str(;)
       end %str(;)

      CALL SYMPUTX("&a.mtext", 
        '%MACRO ' !! "&a." !! '(J,M);' !! 
        '%local J M; %if %qupcase(&M.)= %then %do;' !! /* empty value is output, I is input */
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&&&sysmacroname.&J.%end;' !! 
        '%else %do;' !! IFC(VTYPE(&v.)='N','.','') !! /* put . for numeric out of range */
          '%put WARNING:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Missing value is used.;' !!
        '%end;' !!
        '%end;' !!
        '%else %do; %if %qupcase(&M.)=I %then %do;' !!
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&sysmacroname.&J.%end;' !! 
        '%else %do;' !!
          '%put ERROR:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put ERROR-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
        '%end; %end; %end;' !!
        '%MEND;', 'G') %str(;)
        _ALL_MACROVARIABLES_CREATED_ + &v._h&i..NUM_ITEMS %str(;)
      ;
    %end;  
  %else
    %do;
      /*%put *in hash#*&=a &=v*;*/
      /* toBeCalled is not changed */
      /*%let toBeCalled = %superq(toBeCalled)%str(;) ;*/
      %let toBeLooped 
    = %superq(toBeLooped)%str(;) 
      &v._i&i.+1%str(;)
      /* quote(cats(...)) */
      if "&q." in ('1', "2") then  
       do %str(;)
        CALL SYMPUTX(CATS("&a.", put(&v._i&i., best32.)), quote(cats(&v.), ifc("&q."='1',"'",'"')), 'G') %str(;)
       end %str(;)
      else
       do %str(;)
        CALL SYMPUTX(CATS("&a.", put(&v._i&i., best32.)), &v., 'G') %str(;)
       end %str(;)
      ;
      %let toBeAfter 
    = %superq(toBeAfter)%str(;)
      CALL SYMPUTX("&a.LBOUND",        1, 'G') %str(;)
      CALL SYMPUTX("&a.HBOUND", &v._i&i., 'G') %str(;)
      CALL SYMPUTX("&a.N"     , &v._i&i., 'G') %str(;)
      CALL SYMPUTX("&a.mtext", 
        '%MACRO ' !! "&a." !! '(J,M);' !! 
        '%local J M; %if %qupcase(&M.)= %then %do;' !!  /* empty value is output, I is input */
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then &&&sysmacroname.&J.;' !! 
        '%else %do;' !! IFC(VTYPE(&v.)='N','.','') !! /* put . for numeric out of range */
          '%put WARNING:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
          '%put WARNING-[Macroarray &sysmacroname.] Missing value is used.;' !!
        '%end;' !!
        '%end;' !!
        '%else %do; %if %qupcase(&M.)=I %then %do;' !!
        '%if %sysevalf( (&&&sysmacroname.LBOUND LE &J.) * (&J. LE &&&sysmacroname.HBOUND) ) %then %do;&sysmacroname.&J.%end;' !! 
        '%else %do;' !!
          '%put ERROR:[Macroarray &sysmacroname.] Index &J. out of range.;' !!
          '%put ERROR-[Macroarray &sysmacroname.] Should be between &&&sysmacroname.LBOUND and &&&sysmacroname.HBOUND;' !!
        '%end; %end; %end;' !!
        '%MEND;', 'G') %str(;)
        _ALL_MACROVARIABLES_CREATED_ + &v._i&i. %str(;)
      ;
    %end;
%end;

%let rc = %sysfunc(
dosubl(
/*===============================================================================================*/
options nonotes nosource nomprint ps = min %str(;)
DATA _NULL_ %str(;)
 IF 0 THEN SET &ds. %str(;)
  
 %unquote(&toBeCalled.) %str(;)

 DO UNTIL (EOF) %str(;)
   SET &ds. END=EOF %str(;)
   %unquote(&toBeLooped.) %str(;) /* place where macroarray are created */
 END %str(;)

 %unquote(&toBeAfter.) %str(;)

 CALL SYMPUTX ("_all_macrovariables_created_", _all_macrovariables_created_, "L") %str(;)
 STOP %str(;)
RUN %str(;)
/*===============================================================================================*/
));

%do i = 1 %to &numvars.;
  /* select variable and check if there is different name for array  */
  %let av = %qscan(%superq(vars), &i., %str( )); 
  %let avNumber = %qsysfunc(countw(%superq(av), %str(#|)));
  /*%put *&=av*&=avNumber*;*/
  
  /* get variable name and array name */
  %let v = %qscan(%superq(av),          1, %str(#|));
  %let a = %qscan(%superq(av), &avNumber., %str(#|));
  
  /* compile macro-array */
  %if %superq(macarray) in (Y y) %then
  %do;
    %unquote(&&&a.mtext)
  %end;

  %symdel %unquote(&a.mtext) / NOWARN ;
%end;
%put NOTE:[&sysmacroname.] &_ALL_macrovariables_CREATED_. macrovariables created;
%Exit:
%mend array;


/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################


**EXAMPLE 1.** Basic use-case. 
    Creating macroarray like in the array statement.
    Values not variables names are used by default.
    Different types of brackets are allowed.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(a[*] x1-x5 (1:5))

  %array(b{5} (5*17), q=1)

  %* Mind the $ since it is a character array!;
  %array(c(3) $ 10 ("a A" "b,B" "c;C"))

  %array(d x1-x5 (5 4 3 2 1))
  %put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Index ranges.
    If range starts < 0 then it is shifted to 0.
    In case when range is from `1` to `M`
    then macrovariable `<arrayname>N` is set to `M`
    In case when range is different
    the `<arrayname>N` returns number of 
    elements in the array `(Hbound - Lbound + 1)`.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(d[-2:2] $ ("a" "b" "c" "d" "e"))
  %put &=dLBOUND. &=dHBOUND. &=dN.; 
  %put &=d0. &=d1. &=d2. &=d3. &=d4.;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Functions.
   It is possible to assign value of a function
   or an expression to a cell of the array, 
   e.g. `array[_I_] = function(...)`.
   You can use an iterator in a function.
   As in case of usual arrays it is `_I_`. 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(e[-3:3] $, function = "A" ) 
  %put &=eLBOUND. &=eHBOUND. &=eN.; 
  %put &=e0. &=e1. &=e2. &=e3. &=e4. &=e5. &=e6.;

  %array(f[-3:3], function = (2**_I_) ) 
  %put &=fLBOUND. &=fHBOUND. &=fN.;
  %put &=f0. &=f1. &=f2. &=f3. &=f4. &=f5. &=f6.;

  %array(g[0:2], function = ranuni(123) )
  %put &=gLBOUND. &=gHBOUND. &=gN.;
  %put &=g0. &=g1. &=g2.; 

  %* Or something more complex;
  %array(gg[0:11] $ 11, function = put(intnx("MONTH", '1jun2018'd, _I_, "E"), yymmn.), q=1)
  %put &=ggLBOUND. &=ggHBOUND. &=ggN.;
  %put &=gg0 &=gg1 &=gg2 ... &=gg11;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Functions cont.
    If there is need for set-up something *before* or *after*:

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(h[10:12]
        ,function = rand('Uniform')
        ,before = call streaminit(123) 
        ,after = call sortn(of h[*])
        ) 
  %put &=h10. &=h11. &=h12.; 
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** Fibonacci series. 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(i[1:10] (10*0)
        ,function = ifn(_I_ < 2, 1, sum(i[max(_I_-2,1)], i[max(_I_-1,2)]) ) )
  %put &=i1 &=i2 &=i3 &=i4 &=i5 &=i6 &=i7 &=i8 &=i9 &=i10;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6a.** Quoted "Uppercas Letters"

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(UL[26] $, function = byte(rank("A")+_I_-1) , q=1)
  %put &=UL1 &=UL2 ... &=UL25 &=UL26;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6b.** "Lowercase Letters" 
    Extended by `macarray=Y` option and
    the input mode support (with `I`).

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(ll[26] $, function = byte(rank("a")+_I_-1), macarray=Y) 
  %put *%ll(&llLBOUND.)*%ll(3)*%ll(4)*%ll(5)*...*%ll(25)*%ll(&llHBOUND.)*;

  %* The range handling, warning;
  %put *%ll(265)*;
 
  %* The input mode; 
  %put *before:*%ll(2)*;
  %let %ll(2,I) = bbbbb;
  %put *after: *%ll(2)*;

  %* The range handling, error;
  %let %ll(265,I) = bbb;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 7.** The use of `vnames=Y`

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(R R1978-R1982)
  %put &=R1 &=R2 &=R3 &=R4 &=R5;

  %array(R R1978-R1982 (78:82))
  %put &=R1 &=R2 &=R3 &=R4 &=R5;

  %array(R R1978-R1982 (78:82), vnames=Y)
  %put &=R1 &=R2 &=R3 &=R4 &=R5;

  %array(R R1978-R1982, vnames=Y)
  %put &=R1 &=R2 &=R3 &=R4 &=R5;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 8.** A "no name" array i.e. the `_[*]` array 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(_[*] x1-x5 (1:5))
  %put _user_;

  %array(_[*] p q r s (4*42))
  %put _user_;

  %* If no variables names than use _1 _2 ... _N;
  %array(_[4] (-1 -2 -3 -4)) 
  %put &=_1 &=_2 &=_3 &=_4;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 9.** Pure macro code can be used in a data step.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  data test1;
    set sashelp.class;
    %array(ds[*] d1-d4 (4*17))
    a1 = &ds1.;
    a2 = &ds2.;
    a3 = &ds3.;
    a4 = &ds4.;
  run;

  data test2;
    set sashelp.class;
    %array(_[*] j k l m (4*17))
    a1 = &j.;
    a2 = &k.;
    a3 = &l.;
    a4 = &m.;
  run;

  data test3;
    set sashelp.class;
    %array(alpha[*] j k l m (101 102 103 104), macarray=Y)
    a1 = %alpha(1);
    a2 = %alpha(2);
    a3 = %alpha(3);
    a4 = %alpha(4);
    a5 = %alpha(555);
  run;

  data test4;
    set sashelp.class;
    %array(beta[*] j k l m (101 102 103 104), vnames=Y, macarray=Y)
    a1 = "%beta(1)";
    a2 = "%beta(2)";
    a3 = "%beta(3)";
    a4 = "%beta(4)";
    a5 = "%beta(555)";
  run;

  data test5;
    set sashelp.class;
    %array(gamma[4] $ 12 ("101" "102" "103" "104"), macarray=Y)
    a1 = "%gamma(1)";
    a2 = "%gamma(2)";
    a3 = "%gamma(3)";
    a4 = "%gamma(4)";
    a5 = "%gamma(555)";
  run;

  data test6;
    set sashelp.class;
    %array(ds = sashelp.cars, vars = Cylinders|, macarray=Y)
    a0 = %Cylinders(0);
    a1 = %Cylinders(1);
    a2 = %Cylinders(2);
    a3 = %Cylinders(3);
    a4 = %Cylinders(4);
    a5 = %Cylinders(555);
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 10.** Creating an array from a dataset, basic case.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(ds = sashelp.class, vars = height weight age)
  %put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 11. Creating an array from a dataset, advanced.
    If: `vars = height#h weight weight|w age|`
    then create:
    1. macroarray "h" with ALL(#) values of variable "height"
    2. macroarray "weight" with ALL(no separator is equivalent to #) values of variable "weight"
    3. macroarray "w" with UNIQUE(|) values of variable "weight"
    4. macroarray "age" with UNIQUE(|) values of variable "age"
    Currently the only separator in VARS is a space.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(ds = sashelp.class, vars = height#h weight weight|w age|, q=1)
  %put _user_;

  %array(ds = sashelp.class, vars = height#hght weight weight|wght age|, macarray=Y, q=1)
  %put *%hght(&hghtLBOUND.)**%weight(2)**%wght(&wghtHBOUND.)**%age(3)*;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 12.** Creating an array from a dataset with dataset options 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(ds = sashelp.cars(obs=100 where=(Cylinders=6)), vars = Make| Type| Model, macarray=Y)
  %put *%make(&makeLBOUND.)*%Model(2)*%Model(3)*%Model(4)*%type(&typeHBOUND.)*;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 13.** Creating an array and macro from existing list of macrovariables 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %let myTest3 = 13;
  %let myTest6 = 16;
  %let myTest9 = 19;
  
  %array(myTest, macarray=M, q=1)
  %do_over(myTest, phrase = %nrstr(%put *&_I_.*%myTest(&_I_.)*;))
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since January 2019.                   */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
