/*** HELP START ***//*
 
## >>> `%concatArrays()` macro: <<< <a name="concatarrays-macro"></a> ###########
 
The `%concatArrays()` macro allows to concatenate
two macroarrays created by the `%array()` macro.

By default values of the second macroarray are removed.

Dimensions of the first macroarray are extended.

The `%concatArrays()` macro executes like a pure macro code.

### SYNTAX: #####################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%concatArrays(
  first
 ,second
 <,removeSecond=Y>
) 
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `first` -          *Required*, a name of a macroarray created by the `%array()` macro.

2. `second` -         *Required*, a name of a macroarray created by the `%array()` macro.

*  `removeSecond=Y` - *Optional*, default value `Y`, if set to `Y` then 
                      the second array is removed.

*//*** HELP END ***/

%macro concatArrays(
  first         
, second        
, removeSecond=Y
)
/minoperator
;
%local rc;

%if NOT (%superq(removeSecond) in (Y y)) %then %let removeSecond = N;

%if (NOT %sysevalf(%superq(first)=)) and (NOT %sysevalf(%superq(second)=)) %then
  %do;
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
    %if %SYMEXIST(&first.LBOUND)  and %SYMEXIST(&first.HBOUND)  and %SYMEXIST(&first.N)
    and %SYMEXIST(&second.LBOUND) and %SYMEXIST(&second.HBOUND) and %SYMEXIST(&second.N)
    %then
      %do;
        %if %upcase(&first.) = %upcase(&second.) %then %let removeSecond = N;
        %let rc = %sysfunc(dosubl(
        /*===============================================================================================*/
          options nonotes nosource ps=min %str(;)
          data _null_ %str(;)
            do i = 1 to (&&&second.HBOUND - &&&second.LBOUND + 1) %str(;)
              call symputX( cats("&first.", put(&&&first.HBOUND + i, best32.))
                          , symget( cats("&second.", put(&&&second.LBOUND + (i-1), best32.)) )
                          , "G") %str(;)
              if "&removeSecond." = "Y" then
                call symdel(cats("&second.", put(&&&second.LBOUND + (i-1), best32.)),  "NOWARN") %str(;)
            end %str(;)
            call symputX("&first.HBOUND", put(&&&first.HBOUND + (i-1), best32.), "G") %str(;)
            call symputX("&first.N", put(&&&first.HBOUND + i - &&&first.LBOUND, best32.), "G") %str(;)
            if "&removeSecond." = "Y" then
              do %str(;)
                call symdel("&second.LBOUND",  "NOWARN") %str(;)
                call symdel("&second.HBOUND",  "NOWARN") %str(;)
                call symdel("&second.N"     ,  "NOWARN") %str(;)
              end %str(;)
          run %str(;)
        /*===============================================================================================*/
        ));
      %end;
    %else
      %do;
        %put ERROR:[&SYSMACRONAME.] The First or the Second array is undefined.;
        %put ERROR-[&SYSMACRONAME.] No modification done.;
      %end;
  /*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/  
  %end;
%else
  %do;
    %put WARNING:[&SYSMACRONAME.] The First or the Second argument is missing.;
    %put WARNING-[&SYSMACRONAME.] No modification done.;
  %end;
%mend concatArrays;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Concatenate macroarrays LL and MM.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(ll[2:4] $ 12,
    function = quote(put(today() + 10*_I_, yymmdd10.)),
    macarray=Y
  )
  %array(mm[10:13] $ 12000,
    function = quote(repeat("A",123*_I_)),
    macarray=Y
  )
  %put *%ll(2)*%ll(3)*%ll(4)*;

  %concatArrays(ll, mm); 
  %put *%ll(2)*%ll(3)*%ll(4)*%ll(5)*%ll(6)**%ll(7)*%ll(8)*;

  %put *%mm(10)**%mm(11)*%mm(12)*%mm(13)*;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Error handling.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %concatArrays(ll, )
  %concatArrays(, mm)

  %concatArrays(noExistA, noExistB)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
---

*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since February 2019.                  */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
