/*** HELP START ***//*
 
## >>> `%do_over()` macro: <<< <a name="do-over-macro"></a>######################

The code of the macro was inspired by 
*Ted Clay's* and *David Katz's* macro `%do_over()`.

The `%DO_OVER()` macro allows to iterate over macroarray created with 
the `macarray=Y` parameter of the `%ARRAY()` macro.

The `%do_over()` macro executes like a pure macro code.

### SYNTAX: #####################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%do_over(
  array 
 <,phrase=%nrstr(%&array(&_I_.))>
 <,between=%str( )>
 <,which = >
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `array` - *Required*, indicates a macroarray which metadata (Lbound, Hbouns)
             are to be used to loop in the `%do_over()`

* `phrase=` - *Optional*, Default value `%nrstr(%&array(&_I_.))`, 
              a statement to be called in each iteration 
              of the internal do_over's loop. Loop iterator is `_I_`, 
              if you want to use `_I_` or array name 
              [e.g. `%myArr(&_I_.)`] *enclose it* in the `%NRSTR()` 
              macro quoting function.

* `between=` - *Optional*, default value `%str( )` (space), 
               a statement to be called in between each 
               iteration of the internal do_over loop.
               If macroquoted (e.g. `%str( + )`) then the `%unquote()` 
               function is automatically applied.

* `which=` - *Optional*, a _SPACE_ separated list of indexes which 
             should be used to iterate over selected macroarray.
             Possible special characters are `H` and `L` which means 
             *high* and *low* bound of an array, list could be set with 
             colons(`:`) in form of `start:end:by` (*no spaces between!*), 
             if `by` is omitted the default is `1`. If possible use
             `1:5` rather `1 2 3 4 5` since the firs works faster.

*//*** HELP END ***/

%macro do_over(
  array 
, phrase=%nrstr(%&array(&_I_.))
, between=%str( )
, which = 
);
%local _I_ _wc_ _w_ start end by _first_;
%if %superq(which) = %then 
  %do;
    /* execute the loop for ALL elements */
    %do _I_ = &&&array.LBOUND %to &&&array.HBOUND;
      %if &_I_. NE &&&array.LBOUND %then %do;%unquote(&between.)%end;%unquote(%unquote(&phrase.))
    %end;
  %end;
%else
  %do;
    %do _wc_ = 1 %to %qsysfunc(countw(&which., %str( )));
      %let _w_ = %qscan(&which., &_wc_., %str( ));

      %let start = %qupcase(%qscan(&_w_., 1, :));
      %let end   = %qupcase(%qscan(&_w_., 2, :));
      %let by    = %qupcase(%qscan(&_w_., 3, :));
      
      /* testing conditions for START */
      %if %superq(start) = H %then %let start = &&&array.HBOUND;
      %else 
        %if %superq(start) = L %then %let start = &&&array.LBOUND;

      /* testing conditions for END */
      %if %superq(end) = %then %let end = %sysevalf(&start.);
      %else 
        %if %superq(end) = H %then %let end = &&&array.HBOUND;
        %else 
          %if %superq(end) = L %then %let end = &&&array.LBOUND;

      /* testing conditions for BY */
      %if %superq(by) = %then %let by = 1;

      /* execute the loop over selected elements */
      %do _I_ = &start. %to &end. %by &by.;
        %if &_first_. NE %then %do;%unquote(&between.)%end;%else%do;%let _first_=!;%end;%unquote(%unquote(&phrase.))
      %end;
    %end;
  %end;
%mend do_over;

/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################

**EXAMPLE 1.** Simple looping.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(beta[*] j k l m (101 102 103 104), vnames=Y, macarray=Y)
  
  %put #%do_over(beta)#;

  %put #%do_over(beta, phrase=%nrstr("%beta(&_I_.)"), between=%str(,))#;

  data test1;
    %array(beta[*] j k l m (101 102 103 104), vnames=Y, macarray=Y)
    %do_over(beta, phrase=%nrstr(a&_I_. = "%beta(&_I_.)";))
    put _all_;
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Multiple arrays looping.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(alpha[*] j k l m n, vnames=Y, macarray=Y)
  %array( beta[5] $ , function = "a",  macarray=Y)
  %array(gamma[4] (101 102 103 104),   macarray=Y)

  data test2;
    call streaminit(123);
    %do_over(beta
           , phrase = %nrstr(%beta(&_I_.) = %gamma(&_I_.) * rand('Uniform'); output;) 
           , between = put _all_;
            );
    put _all_;
  run;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Multiple arrays looping, cont.
    Create multiple datasets.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %do_over(beta
  , phrase = %nrstr( 
      data %alpha(&_I_.)2;  
       call streaminit(123); 
       %beta(&_I_.)x = %gamma(&_I_.) * rand('Uniform');  
       output; 
      run; 
  )
  )
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Multiple arrays looping, cont.
    Create multiple datasets using a macro.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %macro doit(ds, var=a, val=1);
    data &ds.; 
     call streaminit(123);
     &var. = &val. * rand('Uniform'); 
     output;
    run;
  %mend doit;

  %do_over(beta
    , phrase = %nrstr( 
      %DOIT(%alpha(&_I_.)1, var = %beta(&_I_.), val = %gamma(&_I_.))
      )
  )
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** `%do_over()` inside `%array()`

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(test[*] x1-x12 (1:12), macarray=Y)

  %put **%test(1)**%test(12)**;

  %put #%do_over(test)#;

  %array(abc[*] x1-x12 (%do_over(test,phrase=%nrstr(%eval(100-%test(&_I_.))))), macarray=Y)

  %put **%abc(1)**%abc(12)**;

  %put #%do_over(abc)#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6.** Looping over array with *macroquoted* separator.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(alpha[11] (5:15), macarray=Y)

  %let x = %do_over(alpha
  , phrase = %NRSTR(%alpha(&_I_.))
  , between= %str( + )
  );
  %put &=x.;
  %put %sysevalf(&x.);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 7.** Working with the `WHICH=` optional parameter

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
  %array(test[*] x01-x12, vnames= Y, macarray=Y)

  %put #%do_over(test)#;

  %put #%do_over(test, which= 1 3 5)#;

  %put #%do_over(test, which= 1:5)#;

  %put #%do_over(test, which= 1:5:2 7 8)#;

  %put #%do_over(test, which= L:H l:h)#;

  %put #%do_over(test, which= L:3 10:h)#;

  %put #%do_over(test, which= L:H h:l:-1 13 14)#;

  %put #%do_over(test, which= %eval(1+1):%eval(5+1))#;

  %put #%do_over(test, which= L:H h:l:-1 13 14, between=%str(,))#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
---
*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since January 2019.                   */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You've been warned! You are using it on your own risk.             */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
