
/*** HELP START ***//*
 
## >>> `%mcDictionary()` macro: <<< <a name="mcdictionary-macro"></a> #######################

The `%mcDictionary()` macro provided in the package 
is designed to facilitate the idea of a "macro dictionary" 
concept, i.e. *a list of macrovariables with common prefix 
and suffixes generated as a hash digest* which allows 
to use values other than integers as indexes.

The `%mcDictionary()` macro allows to generate other macros 
which behaves like a dictionary. See examples below.

The `%mcDictionary()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%mcDictionary(
   H
  <,METHOD>
  <,DS=>
  <,K=Key>
  <,D=Data>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `H`      - *Required*, a dictionary macro name and a declaration/definition, 
              e.g. `mcDictionary(HT)`. It names a macro which is generated by
              the `%mcDictionary()` macro. Provided name cannot be empty 
              or an underscore (`_`). No longer than *13* characters.

2. `METHOD` - *Optional*, if empty (or DECLARE or DCL) then the code of 
              a macro dictionary is compiled.
              If `DELETE` then the macro dictionary named by `H` and all 
              macrovariables named like "`&H._`" are deleted.

* `DS=`     - *Optional*, if NOT empty then the `&DS.` dataset is used to 
              populate dictionary with keys from variable `&K.` and data
              from variable `&D.` Works only during declaration.

* `K=`      - *Optional*, if the `&DS.` is NOT empty then `&K.` holds a name of
              a variable which keeps or an expression which generates keys values. 
              Default is `Key`.

* `D=`      - *Optional*, if the `&DS.` is NOT empty then `&D.` holds a name of
              a variable which keeps or an expression which generates data values. 
              Default is `Data`.

---

### THE CREATED MACRO `%&H.()`: ####################################################

The created macro imitates behaviour of a dictionary. 

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%&H.(
  METHOD
  <,KEY=> 
  <,DATA=>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `METHOD` - *Required*, indicate what behaviour should be executed. 
              Allowed values are:
                - `ADD`, adds key and data portion to the macro dictionary,
                   *multiple data portions* are NOT available for one key.
                - `FIND`, tests if given key exists in the macro dictionary
                   and, if yes, returns data value associated with the key.
                   For multiple data portions see the `data=` parameter.
                - `CHECK`, returns indicator if the key exists in dictionary.
                - `DEL`, removes key and data portion from the macro dictionary.
                - `LIST`, prints out a dictionary to the log. 
                - `CLEAR` removes all data and keys values.

* `KEY=`    - *Optional*, provides key value for `ADD`, `FIND`, `CHECK`
              and `DEL` methods. 
              Leading and trimming spaces are removed from the value.
              The `MD5(...)` function is used to generate the hash.
              Default value is `_`.

* `DATA=`   - *Optional*, provides data value for the `ADD` method. 
              Default value is blank.


When macro is executed and when data are added the following types of 
*global* macrovariables are created:
- `&H._########_K`,
- `&H._########_V`,
- `&H._KEYSNUM`.

The `#` represents value generated by the `MD5(...)` function for the given key.

The first type keeps information about the key.

The second type keeps information about the value of a given key

The third type keeps the number of unique values of the key.

See examples below to see use cases.

---
*//*** HELP END ***/

%macro mcDictionary(
   H
  ,METHOD
  ,DS=
  ,K=Key
  ,D=Data
)/minoperator;
%if %superq(H) = _ %then
  %do;
    %put WARNING: |----------------------------------------------------|;
    %put WARNING- | The name of Macro dictionary cannot be underscore. |;
    %put WARNING- |----------------------------------------------------|;
    %return;
  %end;

%if %superq(H) = %then
  %do;
    %put ERROR: |-------------------------------------------------|;
    %put ERROR- | The name of Macro dictionary cannot be missing. |;
    %put ERROR- |-------------------------------------------------|;
    %return;
  %end;

%if %length(&H.) > 13 %then
  %do;
    %put ERROR: |--------------------------------------------|;
    %put ERROR- | The length of the name of Macro dictionary |;
    %put ERROR- | cannot be greater than 13.                 |;
    %put ERROR- |--------------------------------------------|;
    %return;
  %end;

%local mtext HASH;
%let mtext = _%sysfunc(datetime(),hex16.)%sysfunc(datetime(),hex8.)_;
/*%put &=mtext.;*/
%let HASH = MD5;

%if (%Qupcase(&METHOD.) = ) or (%Qupcase(&METHOD.) = DECLARE) or (%Qupcase(&METHOD.) = DCL) %then
%do;
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
%let rc = %sysfunc(
dosubl(
/*=========================================================================================================*/
options ps=min nonotes nosource %str(;)
DATA _NULL_ %str(;)
CALL SYMPUTX("&mtext.",
'%MACRO ' !! "&H.(METHOD, KEY=_, DATA=)/minoperator;"                    !! 
/* The first approach was to use hidden macrovariable referring to the dictionary name
   and to call it dynamically by &H. in the code. The second approach was to replace, 
   in the text, all occurrences of `&H.` by `' !! "&H." !! '` to make the created code 
   static. No improvement in the performance was gained but the code became unreadable. 
   Return to the firs approach was recommended.
*/ 
'  %if %Qupcase(&METHOD.) = %then                                            ' !!
'    %do;                                                                    ' !!
'      %put |--------------------------------------|;                        ' !!
'      %put | The following methods are available: |;                        ' !!
'      %put |--------------------------------------|;                        ' !!
'      %put | 1. ADD [A]                           |;                        ' !!
'      %put | 2. FIND [F]                          |;                        ' !!
'      %put | 3. CHECK [C]                         |;                        ' !!
'      %put | 4. DEL [D]                           |;                        ' !!
'      %put | 5. LIST [L]                          |;                        ' !!
'      %put | 6. CLEAR                             |;                        ' !!
'      %put |--------------------------------------|;                        ' !!
'      %return;                                                              ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %local H hash _newkey_ ;                                                  ' !!
'  %let H = ' !! "&H." !! ';                                                 ' !! /* !!!! */
ifc(upcase("&HASH.")="MD5",
'  %let hash = %sysfunc(MD5(%superq(KEY)),hex16.);                           ', ' ') !! /* &hash. */
/* *** */
'  %global &H._KEYSNUM;                                                      ' !!
'  %if %Qupcase(&METHOD.) in (ADD A) %then                                   ' !!
'    %do;                                                                    ' !!
'      %let _newkey_ = 1;                                                    ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash._K )) %then                      ' !!
'        %do;                                                                ' !!
'          %let _newkey_ = 0;                                                ' !!
'          %if not(%superq(KEY) = %superq(&H._&hash._K)) %then               ' !!
'            %do;                                                            ' !!
'              %put WARNING: HASH &hash. is a collision for keys:;           ' !!
'              %put WARNING- %superq(KEY);                                   ' !!
'              %put WARNING- %superq(&H._&hash._K));                         ' !!
'            %end;                                                           ' !!
'        %end;                                                               ' !!
'      %global &H._&hash._V &H._&hash._K;                                    ' !!
'      %let &H._&hash._V = %superq(DATA);                                    ' !!
'      %let &H._&hash._K = %superq(KEY); /*key value */                      ' !!
'      %let &H._KEYSNUM = %eval(&&&H._KEYSNUM + &_newkey_.);                 ' !!
'      %return;                                                              ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %if %Qupcase(&METHOD.) in (FIND F) %then                                  ' !!
'    %do;                                                                    ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash._K )) %then                      ' !!
'        %do;                                                                ' !!
'          %if not(%superq(KEY) = %superq(&H._&hash._K)) %then               ' !!
'            %do;                                                            ' !!
'              %put WARNING: HASH &hash. is a collision for keys:;           ' !!
'              %put WARNING- %superq(KEY);                                   ' !!
'              %put WARNING- %superq(&H._&hash._K));                         ' !!
'            %end;                                                           ' !!
'          %do;%unquote(&&&H._&hash._V)%end;                                 ' !!
'          %return;                                                          ' !!
'        %end;                                                               ' !!
'      %else %return;                                                        ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %if %Qupcase(&METHOD.) in (CHECK C) %then                                 ' !!
'    %do;                                                                    ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash._K )) %then                      ' !!
'        %do;                                                                ' !!
'          %if not(%superq(KEY) = %superq(&H._&hash._K)) %then               ' !!
'            %do;                                                            ' !!
'              %put WARNING: HASH &hash. is a collision for keys:;           ' !!
'              %put WARNING- %superq(KEY);                                   ' !!
'              %put WARNING- %superq(&H._&hash._K));                         ' !!
'            %end;                                                           ' !!
'          %do;1%end;                                                        ' !!
'          %return;                                                          ' !!
'        %end;                                                               ' !!
'      %else %do;0%return;%end;                                              ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %if %Qupcase(&METHOD.) in (DEL D) %then                                   ' !!
'    %do;                                                                    ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash._K )) %then                      ' !!
'        %do;                                                                ' !!
'          %let _newkey_ = -1;                                               ' !!
'          %if not(%superq(KEY) = %superq(&H._&hash._K)) %then               ' !!
'            %do;                                                            ' !!
'              %put WARNING: HASH &hash. is a collision for keys:;           ' !!
'              %put WARNING- %superq(KEY);                                   ' !!
'              %put WARNING- %superq(&H._&hash._K));                         ' !!
'            %end;                                                           ' !!
'          %SYMDEL &H._&hash._K &H._&hash._V / NOWARN;                       ' !!
'          %let &H._KEYSNUM = %eval(&&&H._KEYSNUM + &_newkey_.);             ' !!
'        %end;                                                               ' !!
'      %return;                                                              ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %if %Qupcase(&METHOD.) in (LIST L) %then                                  ' !!
'    %do;                                                                    ' !!
'      %let hash = %sysfunc(MD5(&H.),hex16.);                                ' !!
'      %let i = %sysfunc(doSubL(                                             ' !!
'      options nonotes nosource nomprint nosymbolgen ps=min ls=max %str(;)   ' !!
'        proc sql noprint %str(;)                                            ' !!
'          create table work._TEMP_&hash. as                                 ' !!
'          select scan(name,2,"_") as hash length 16                         ' !!
'          from dictionary.macros                                            ' !!
"          where (                                                           " !! /* !!!! */
"              name like '%upcase(&H.)\_________________\_K' escape '\'      " !! /* !!!! */
"              )                                                             " !! /* !!!! */
'            and scope = "GLOBAL"                                            ' !!
'          order by hash                                                     ' !!
'          %str(;)                                                           ' !!
'        quit %str(;)                                                        ' !!
'        data _null_ %str(;)                                                 ' !!
'          if 1=_N_ then put "Content of the dictionary: " / %str(;)         ' !!
'          if 1=_E_ then put "-------------------------- " / %str(;)         ' !!
'          set work._TEMP_&hash. end=_E_ %str(;)                            ' !!
'            length key val $ 32767 %str(;)                                  ' !!
'            key = symget("%upcase(&H.)_" !! hash !! "_K") %str(;)           ' !!
'            val = symget("%upcase(&H.)_" !! hash !! "_V") %str(;)           ' !!
'            put key= / hash= / val= / %str(;)                               ' !!
'        run %str(;)                                                         ' !!
'        proc delete data = work._TEMP_&hash. %str(;)                        ' !!
'        run %str(;)                                                         ' !!
'      ));                                                                   ' !!
'    %end;                                                                   ' !!
'                                                                            ' !!
'  %if %Qupcase(&METHOD.) = CLEAR %then                                      ' !!
'    %do;                                                                    ' !!
'      %let hash = %sysfunc(MD5(&H.),hex16.);                                ' !!
'      %let i = %sysfunc(doSubL(                                             ' !!
'        options nonotes nosource nomprint nosymbolgen %str(;)               ' !!
'        proc sql noprint %str(;)                                            ' !!
'          create table work._TEMP_&hash. as                                 ' !!
'          select name                                                       ' !!
'          from dictionary.macros                                            ' !!
"          where (                                                           " !! /* !!!! */
"              name like '%upcase(&H.)\_________________\_K' escape '\'      " !! /* !!!! */
"              or                                                            " !! /* !!!! */
"              name like '%upcase(&H.)\_________________\_V' escape '\'      " !! /* !!!! */
"              )                                                             " !! /* !!!! */
'            and scope = "GLOBAL"                                            ' !!
'          %str(;)                                                           ' !!
'        quit %str(;)                                                        ' !!
'        data _null_ %str(;)                                                 ' !!
'            set work._TEMP_&hash. %str(;)                                   ' !!
'            call execute(''%symdel '' !! name !! '' / nowarn;'') %str(;)    ' !!
'        run %str(;)                                                         ' !!
'        proc delete data = work._TEMP_&hash. %str(;)                        ' !!
'        run %str(;)                                                         ' !!
'      ));                                                                   ' !!
'      %let &H._KEYSNUM = 0;                                                 ' !!
'    %end;                                                                   ' !!
'%MEND;', 'G') %str(;)
RUN %str(;)
/*=========================================================================================================*/
));
&&&mtext.
%symdel &mtext. / NOWARN ;
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
%if NOT (%superq(DS) = ) %then
  %do;
    %if %sysfunc(exist(%scan(%superq(DS),1,()))) %then
      %do;
        %put NOTE:[&sysmacroname.] Populating dictionary %superq(H).;
        %local R; /* random string from date */
        %let R  = %sysfunc(datetime(),hex16.);
        %local _raw_;
        %if %superq(K) = %then %let K=" ";
        %if %superq(D) = %then %let D=" ";
        %global &H._KEYSNUM;
        %let rc = %sysfunc(dosubl(%str(
        /*=========================================================================================================*/
        options ps=min nonotes nosource ;
        data _null_;
          _N_=0;
          __&R.__KEYSNUM = 0;
          if nobs then
            do until(eof);
              _N_+1;
              set &DS. end=eof nobs=nobs;
              __&R.__K = &K.;
              __&R.__D = &D.;

              __&R.__hash = put(MD5(strip(__&R.__K)),hex16.);
              __&R.__newKey = (0 = symglobl( cats("&H._",__&R.__hash,"_K") ));

              call symputX("_raw_",__&R.__K,'L');
              call symputX(cats("&H._",__&R.__hash,"_K"),resolve('%superq(_raw_)'),'G');
              call symputX(cats("&H._",__&R.__hash,"_V"),__&R.__D,'G');

              __&R.__KEYSNUM + __&R.__newKey;
              /*put __&R.__KEYSNUM=;*/
              call symputX("&H._KEYSNUM",__&R.__KEYSNUM,'G');

            end;
          stop;
        run;
        /*=========================================================================================================*/
        )));
        %symdel R / nowarn;
        %put NOTE:[&sysmacroname.] -----------------------------;
      %end;
    %else
      %do;
        %put WARNING:[&sysmacroname.] Dataset %superq(DS) does not exist!;
      %end; 
  %end;
%return;
%end;

%if %Qupcase(&METHOD.) = DELETE %then
  %do;
    %let _end_ = %sysfunc(doSubL(
      options NOnotes NOsource NOmprint NOsymbolgen %str(;)
      proc sql noprint %str(;)
        create table work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) as
        select name
        from dictionary.macros
        where (
            name like "%upcase(&H.)\_________________\_K" !! '%' escape '\'
            or
            name like "%upcase(&H.)\_________________\_V" !! '%' escape '\'
            or
            name like "%upcase(&H.)\_KEYSNUM" !! '%' escape '\'
           )
          and scope = "GLOBAL"
        %str(;)
      quit %str(;)
      data _null_ %str(;)
        set work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) %str(;)

      call execute('%symdel ' !! name !! ' / nowarn %str(;)') %str(;)
      run %str(;)
      proc delete data = work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) %str(;)
      run %str(;)
    ));
    %SYSMACDELETE &H./ NOWARN;
    %return;
  %end;

%put |------------------------------------------|; 
%put | Only the following values are available: |; 
%put |------------------------------------------|; 
%put | 1. a blank value or DECLARE or DCL       |; 
%put | 2. DELETE                                |; 
%put |------------------------------------------|; 
%mend mcDictionary;


/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################


**EXAMPLE 1.** Basic use-case. 
  Creating macro dictionary, macro `Dict` is generated.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcDictionary(Dict)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Add elements to the `Dict`.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%Dict(ADD,key=x,data=17)
%Dict(ADD,key=y y,data=42)
%Dict(ADD,key=z z z,data=303)

%put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Add some duplicates for the key x. 
  See macrovariables created.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%Dict(ADD,key=x,data=18) 

%put _user_;

%Dict(ADD,key=x,data=19)

%put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Check for the key `x` and non existing key `t`.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put ##%Dict(CHECK,key=x)##;
%put ##%Dict(CHECK,key=t)##;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Prints data values for various keys.
  Key `t` does not exist in the macrodictionary.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put #%Dict(FIND,key=x)#;
%put #%Dict(FIND,key=y y)#;
%put #%Dict(FIND,key=z z z)#;
%put #%Dict(FIND,key=t)#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  List dictionary content to the log.
  
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%Dict(LIST);
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Delete keys.
  Key `t` does not exist in the macrodictionary.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put #%Dict(DEL,key=z z z)#;
%put _user_;
%put #%Dict(DEL,key=t)#;
%put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Clear and delete macro dictionary `Dict`.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%Dict(CLEAR)
%put _user_;

%mcDictionary(Dict,DELETE)
%put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2A.** Populate macro dictionary from a dataset "by hand".

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcDictionary(CLASS)
%let t = %sysfunc(datetime());
data _null_;
  set sashelp.class;
  call execute('%CLASS(ADD,key=' !! name !! ',data=' !! age !! ')');
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=Class_KEYSNUM.;
%put _user_;
%CLASS(CLEAR)


%mcDictionary(CARS)
%let t = %sysfunc(datetime());
data _null_;
  set sashelp.cars(obs=42);
  call execute('%CARS(ADD,key=' !! catx("|",make,model,type)  !! ',data=' !! put(MPG_CITY*10,dollar10.2) !! ')');
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=CARS_KEYSNUM.;
%CARS(LIST);

%put %CARS(F,key=Audi|TT 3.2 coupe 2dr (convertible)|Sports);

%CARS(CLEAR)
%put &=CARS_KEYSNUM.;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2B.** Populate macro dictionary from a dataset "automatically".

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%let t = %sysfunc(datetime());
%mcDictionary(CLASS,DCL,DS=sashelp.class,k=name,d=_N_)
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=CLASS_KEYSNUM.;
%put _user_;
%CLASS(CLEAR)


%let t = %sysfunc(datetime());
%mcDictionary(CARS,DCL,DS=sashelp.cars(obs=42),k=catx("|",make,model,type),d=put(MPG_CITY*10,dollar10.2))
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=CARS_KEYSNUM.;
%CARS(LIST);

%put %CARS(F,key=Audi|TT 3.2 coupe 2dr (convertible)|Sports);

%CARS(CLEAR)
%put &=CARS_KEYSNUM.;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Data portion may require quoting and un-quoting.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcDictionary(CODE)
%CODE(CLEAR)
%CODE(ADD,key=data, data=%str(data test; x = 42; run;))
%CODE(ADD,key=proc, data=%str(proc print; run;))
%CODE(ADD,key=macro,data=%nrstr(%put *1*2*3*4*;))

%CODE(FIND,key=data)
%CODE(FIND,key=proc) 
%unquote(%CODE(FIND,key=macro))

%CODE(LIST);

%mcDictionary(CODE,DELETE)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Longer lists.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%let size = 1000;

%mcDictionary(AAA)

%let t = %sysfunc(datetime());
data _null_;
  do i = 1 to &size.;
    call execute(cats('%AAA(ADD,key=A', i, ',data=', i, ')'));
  end;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put %AAA(F,key=A555) %AAA(CHECK,key=A555);
%put &=AAA_KEYSNUM;
%AAA(CLEAR)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** Forbidden names.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcDictionary()
%mcDictionary(_)

%mcDictionary(ABCDEFGHIJKLMN) %* bad;
%mcDictionary(ABCDEFGHIJKLM)  %* good;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6.** More fun with datasets.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas

data work.metadata;
  input key :$16. data :$128.;
cards;
ID ABC-123-XYZ
path /path/to/study/data
cutoffDT 2023-01-01
startDT 2020-01-01
endDT 2024-12-31
MedDRA v26.0
;
run;
proc print;
run;

%mcDictionary(Study,dcl,DS=work.metadata)

%put _user_;

%put *%Study(F,key=ID)**%Study(C,key=ID)*;

title1 "Study %Study(F,key=ID) is located at %Study(F,key=path)";
title2 "it starts %Study(F,key=startDT) and ends %Study(F,key=endDT)";
footnote "MedDRA version: %Study(F,key=MedDRA)";

proc print data=sashelp.class(obs=7);
run;

title; 
footnote;

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~



---
*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since October 2020.                   */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You have been warned! You are using it on your own risk.           */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
