
/*** HELP START ***//*
 
## >>> `%mcHashTable()` macro: <<< <a name="mchashtable-macro"></a> #######################

The `%mcHashTable()` macro provided in the package 
is designed to facilitate the idea of a "macro hash table" 
concept, i.e. *a list of macrovariables with common prefix 
and suffixes generated as a hash digest* which allows 
to use values other than integers as indexes.

The `%mcHashTable()` macro allows to generate other macros 
which behaves like hash tables or dictionaries. See examples below.

The `%mcHashTable()` macro executes like a pure macro code.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(
   H
  <,METHOD>
  <,HASH=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `H`      - *Required*, a hash table macro name and a declaration/definition, 
              e.g. `mcHashTable(HT)`. It names a macro which is generated by
              the `%mcHashTable()` macro. Provided name cannot be empty 
              or an underscore (`_`). No longer than *10* characters.

2. `METHOD` - *Optional*, if empty (or DECLARE or DCL) then the code of 
              a macro hash table is compiled.
              If `DELETE` then the macro hash table named by `H` and all 
              macrovariables named like "`&H._`" are deleted.

* `HASH=`   - *Optional*, indicates which hashing algorithms should be used,
              available values are `CRC32` or `MD5`, the `CRC32` is the default. 

---

### THE CREATED MACRO `%&H.()`: ####################################################

The created macro imitates behaviour of a hash table or a dictionary. 
It is *not* dedicated for "long-ish" lists (above 1000 elements) since 
the performance may be poor.

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%&H.(
  METHOD
  <,KEY=> 
  <,DATA=>
)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `METHOD` - *Required*, indicate what behaviour should be executed. 
              Allowed values are:
                - `ADD`, adds key and data portion to the macro hash table,
                   *multiple data portions* are available for one key.
                - `FIND`, tests if given key exists in the macro hash table
                   and, if yes, returns data value associated with the key.
                   For multiple data portions see the `data=` parameter.
                - `DP` (data portion) or `CHECK`, returns the number of data
                   portions for a given key.
                - `CLEAR` removes all data and keys values.
                - `KEYIDX`, allows to get data by the key index rather than value.
                - `KEYVAL`, returns key value for a given key index.
                - `CHECKIDX`, returns the number of data portions for 
                   a given key index. 

* `KEY=`    - *Optional*, provides key value for `ADD`, `FIND`,`DP`, `CHECK`
              `CHECKIDX`, `KEYIDX`, and `KEYVAL` methods. Leading and trimming 
              spaces are removed from the value.
              The `hashing(CRC32,...)` function or the `MD5(...)` function is 
              used to generate the hash.

* `DATA=`   - *Optional*, provides data value for the `ADD` method and 
              for the`FIND` method provides data portion number to be 
              extracted. Default value is `1` (used by the `FIND` method).


When macro is executed and when data are added the following types of 
*global* macrovariables are created:
- `&H._########`,
- `&H._########_Xk`,
- `&H._########_Xi`,
- `&H._########_Xi_j`,
- `&H._KEYNUM`,
- and `&H._KEY_i`.

The `#` represents value generated by the `hashing(CRC32,...)` function 
or the `MD5(...)` function for the given key.

The first type keeps information about possible collision for the key.

The second type keeps information about value of a given key, 
the `X` keeps the track of other colliding keys.

The third type keeps information about number of data portions 
for given key, the `X` keeps the track of other colliding keys.

The fourth type keeps the data portion, the `j` indicates data portion number.

The fifth type keeps the number of unique values of the key.

The sixth type keeps the list of unique values of the key, 
the `i` indicates key number.

See examples below to see use cases.

---
*//*** HELP END ***/

%macro mcHashTable(H,METHOD,HASH=CRC32)/minoperator;
%if %superq(H) = _ %then
  %do;
    %put WARNING: |----------------------------------------------------|;
    %put WARNING- | The name of Macro Hash Table cannot be underscore. |;
    %put WARNING- |----------------------------------------------------|;
    %return;
  %end;

%if %superq(H) = %then
  %do;
    %put ERROR: |-------------------------------------------------|;
    %put ERROR- | The name of Macro Hash Table cannot be missing. |;
    %put ERROR- |-------------------------------------------------|;
    %return;
  %end;

%if %length(&H.) > 10 %then
  %do;
    %put ERROR: |-------------------------------------------|;
    %put ERROR- | The length of th name of Macro Hash Table |;
    %put ERROR- | cannot be greather than 10.               |;
    %put ERROR- |-------------------------------------------|;
    %return;
  %end;

%if not(%Qupcase(&HASH.) in (CRC32 MD5)) %then
  %do;
    %put ERROR: |-----------------------------------------|;
    %put ERROR- | The only available hash algorithms are: |;
    %put ERROR- | - CRC32                                 |;
    %put ERROR- | - MD5                                   |;
    %put ERROR- |-----------------------------------------|;
    %return;
  %end;

%local mtext ;
%let mtext = _%sysfunc(datetime(),hex16.)%sysfunc(datetime(),hex8.)_;
/*%put &=mtext.;*/

%if (%Qupcase(&METHOD.) = ) or (%Qupcase(&METHOD.) = DECLARE) or (%Qupcase(&METHOD.) = DCL) %then
%do;
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
%let rc = %sysfunc(
dosubl(
/*=========================================================================================================*/
options nonotes nosource %str(;)
DATA _NULL_ %str(;)
CALL SYMPUTX("&mtext.",
'%MACRO ' !! "&H.(METHOD, KEY=_, DATA=1)/minoperator;"                    !! 
/* The first approach was to use hidden macrovariable referring to the hash table name
   and to call it dynamically by &H. in the code. The second approach was to replace, 
   in the text, all occurrences of `&H.` by `' !! "&H." !! '` to make the created code 
   static. No improvement in the performance was gained but the code became unreadable. 
   Return to the firs approach was recommended.
*/ 
'  %if %Qupcase(&METHOD.) = %then                                                      ' !!
'    %do;                                                                              ' !!
'      %put |--------------------------------------|;                                  ' !!
'      %put | The following methods are available: |;                                  ' !!
'      %put |--------------------------------------|;                                  ' !!
'      %put | 1. ADD                               |;                                  ' !!
'      %put | 2. FIND                              |;                                  ' !!
'      %put | 3. DP, CHECK                         |;                                  ' !!
'      %put | 4. CLEAR                             |;                                  ' !!
'      %put | 5. KEYIDX                            |;                                  ' !!
'      %put | 6. KEYVAL                            |;                                  ' !!
'      %put | 7. CHECKIDX                          |;                                  ' !!
'      %put |--------------------------------------|;                                  ' !!
'      %return;                                                                        ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %local H hash hash_i hash_k _newkey_ i j /*NO*/;                                    ' !!
'  %let H = ' !! "&H." !! ';                                                           ' !! /* !!!! */
/* select hashing algorithm */
ifc(upcase("&HASH.")="CRC32",
'  %let hash = %sysfunc(hashing(CRC32,&KEY.));                                         ', ' ') !! /* &hash. */
ifc(upcase("&HASH.")="MD5",
'  %let hash = %sysfunc(MD5(&KEY.),hex16.);                                            ', ' ') !! /* &hash. */
/* *** */
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = ADD %then                                                  ' !!
'    %do;                                                                              ' !!
'      %let _newkey_ = 1;                                                              ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash. )) %then                                  ' !!
'        %do;                                                                          ' !!
'          /* loop over keys we have, if found - update, in not found - add new one */ ' !!
'          %do i = 0 %to &&&H._&hash.;                                                 ' !!
'            %let hash_i = &hash._&i.i; /* &hash_i. */                                 ' !!
'            %let hash_k = &hash._&i.k; /* &hash_k. */                                 ' !!
'            %if %superq(KEY) = %superq(&H._&hash_k.) %then                            ' !!
'              %do;                                                                    ' !!
'                %let j = %eval(&&&H._&hash_i. + 1);                                   ' !!
'                %let _newkey_ = 0;                                                    ' !!
'                %goto LeaveLoop;                                                      ' !!
'              %end;                                                                   ' !!
'          %end;                                                                       ' !!
'          %let j = 1;                                                                 ' !!
'        %end;                                                                         ' !!
'      %else                                                                           ' !!
'        %do;                                                                          ' !!
'          %let i = 0;                                                                 ' !!
'          %let j = 1;                                                                 ' !!
'          %global                                                                     ' !!
'            &H._&hash. /* for number of colisions */                                  ' !!
'            &H._KEYSNUM                                                               ' !!
'          ;                                                                           ' !!
'        %end;                                                                         ' !!
'      %LeaveLoop:                                                                     ' !!
'                                                                                      ' !!
'      %let hash_i = &hash._&i.i; /* &hash_i. */                                       ' !!
'      %let hash_k = &hash._&i.k; /* &hash_i. */                                       ' !!
'      %global                                                                         ' !!
'          &H._&hash_i. /* for number of data portions */                              ' !!
'          &H._&hash_i._&j. /* for data */                                             ' !!
'        ;                                                                             ' !!
'      %let &H._&hash. = &i.; /* number of colisions */                                ' !!
'      %let &H._&hash_i. = &j.; /* number of data portions */                          ' !!
'      %let &H._&hash_i._&j. = %superq(DATA);                                          ' !!
'      %if &_newkey_. = 1 %then                                                        ' !!
'        %do;                                                                          ' !!
'          %let &H._KEYSNUM = %eval(&&&H._KEYSNUM + 1);                                ' !!
'          %global &H._KEY_&&&H._KEYSNUM                                               ' !!
'                  &H._&hash_k.; /* for key value */                                   ' !!
'          %let &H._&hash_k. = %superq(KEY); /*key valu */                             ' !!
'          %let &H._KEY_&&&H._KEYSNUM = &hash._&i.;                                    ' !!
'        %end;                                                                         ' !!
'      %return;                                                                        ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = FIND %then                                                 ' !!
'    %do;                                                                              ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash. )) %then                                  ' !!
'        %do;                                                                          ' !!
'          %do i = 0 %to &&&H._&hash.;                                                 ' !!
'            %let hash_i = &hash._&i.i; /* &hash_i. */                                 ' !!
'            %let hash_k = &hash._&i.k; /* &hash_k. */                                 ' !!
'            %if %superq(KEY) = %superq(&H._&hash_k.) %then                            ' !!
'              %do;                                                                    ' !!
'                %if 1 <= %superq(DATA) AND %superq(DATA) <= &&&H._&hash_i. %then      ' !!
'                  %do;%unquote(&&%unquote(&H._&hash_i._%superq(DATA)))%end;           ' !!
'                %return;                                                              ' !!
'              %end;                                                                   ' !!
'            %else %return;                                                            ' !!
'          %end;                                                                       ' !!
'        %end;                                                                         ' !!
'      %else %return;                                                                  ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) in (DP CHECK) %then                                          ' !!
'    %do;                                                                              ' !!
'      %if 1 = %sysfunc(symglobl( &H._&hash. )) %then                                  ' !!
'        %do;                                                                          ' !!
'          %do i = 0 %to &&&H._&hash.;                                                 ' !!
'            %let hash_i = &hash._&i.i; /* &hash_i. */                                 ' !!
'            %let hash_k = &hash._&i.k; /* &hash_k. */                                 ' !!
'            %if %superq(KEY) = %superq(&H._&hash_k.) %then                            ' !!
'              %do;%unquote(&&%unquote(&H._&hash_i.))%return;%end;                     ' !!
'            %else                                                                     ' !!
'              %do;0%return;%end;                                                      ' !!
'          %end;                                                                       ' !!
'        %end;                                                                         ' !!
'      %else%do;0%return;%end;                                                         ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = CLEAR %then                                                ' !!
'    %do;                                                                              ' !!
'      %let hash = %sysfunc(MD5(&H.),hex16.);                                          ' !!
'      %let i = %sysfunc(doSubL(                                                       ' !!
'        options nonotes nosource nomprint nosymbolgen %str(;)                         ' !!
'        proc sql noprint %str(;)                                                      ' !!
'          create table work._TEMP_&hash. as                                           ' !!
'          select name                                                                 ' !!
'          from dictionary.macros                                                      ' !!
"          where name like '%upcase(&H.)\_%' escape '\'                                " !! /* !!!! */
'            and scope = "GLOBAL"                                                      ' !!
'          %str(;)                                                                     ' !!
'        quit %str(;)                                                                  ' !!
'        data _null_ %str(;)                                                           ' !!
'            set work._TEMP_&hash. %str(;)                                             ' !!
'            call execute(''%symdel '' !! name !! '' / nowarn;'') %str(;)              ' !!
'        run %str(;)                                                                   ' !!
'        proc delete data = work._TEMP_&hash. %str(;)                                  ' !!
'        run %str(;)                                                                   ' !!
'      ));                                                                             ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = KEYIDX %then                                               ' !!
'    %do;                                                                              ' !!
'      %if (1 <= &KEY.) AND (&KEY. <= &&&H._KEYSNUM) %then                             ' !!
'        %do;                                                                          ' !!
'          %if (1 <= &DATA.) AND (&DATA. <= &&&&&H._&&&H._KEY_&KEY..i) %then           ' !!
'            %do;%let DATA = &H._&&&H._KEY_&KEY..i_&DATA.;&&&DATA.%return;%end;        ' !!
/*                            HT_&HT_KEY_1.i_1                                           */
/*                            HT_########_0i_1                                           */
'          %else %return;                                                              ' !!
'        %end;                                                                         ' !!
'      %else %return;                                                                  ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = CHECKIDX %then                                             ' !!
'    %do;                                                                              ' !!
'      %if (1 <= &KEY.) AND (&KEY. <= &&&H._KEYSNUM) %then                             ' !!
'        %do;&&&&&H._&&&H._KEY_&KEY..i%return;                                         ' !!
/*              &&HT_&HT_KEY_1.i                                                         */
/*               &HT_########_0i                                                         */
/*                value                                                                  */
'        %end;                                                                         ' !!
'      %else%do;0%return;%end;                                                         ' !!
'    %end;                                                                             ' !!
'                                                                                      ' !!
'  %if %Qupcase(&METHOD.) = KEYVAL %then                                               ' !!
'    %do;                                                                              ' !!
'      %if (1 <= &KEY.) AND (&KEY. <= &&&H._KEYSNUM) %then                             ' !!
'        %do;&&&&&H._&&&H._KEY_&KEY..k%return;                                         ' !!
/*              &&HT_&HT_KEY_1.k                                                         */
/*               &HT_########_0k                                                         */
/*                keyValue                                                               */
'        %end;                                                                         ' !!
'      %else %return;                                                                  ' !!
'    %end;                                                                             ' !!
'%MEND;', 'G') %str(;)
RUN %str(;)
/*=========================================================================================================*/
));
&&&mtext.
%symdel &mtext. / NOWARN ;
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
%return;
%end;

%if %Qupcase(&METHOD.) = DELETE %then
  %do;
    %let _end_ = %sysfunc(doSubL(
      options NOnotes NOsource NOmprint NOsymbolgen %str(;)
      proc sql noprint %str(;)
        create table work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) as
        select name
        from dictionary.macros
        where name like "%upcase(&H.)\_" !! '%' escape '\'
          and scope = "GLOBAL"
        %str(;)
      quit %str(;)
      data _null_ %str(;)
        set work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) %str(;)

      call execute('%symdel ' !! name !! ' / nowarn %str(;)') %str(;)
      run %str(;)
      proc delete data = work._TEMP_%unquote(%sysfunc(MD5(&H.),hex16.)) %str(;)
      run %str(;)
    ));
    %SYSMACDELETE &H./ NOWARN;
    %return;
  %end;

%put |------------------------------------------|; 
%put | Only the following values are available: |; 
%put |------------------------------------------|; 
%put | 1. a blank value or DECLARE or DCL       |; 
%put | 2. DELETE                                |; 
%put |------------------------------------------|; 
%mend mcHashTable;


/*** HELP START ***//*
 
### EXAMPLES AND USECASES: ####################################################


**EXAMPLE 1.** Basic use-case. 
  Creating macro hash table, macro `HT` is generated.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(HT)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Add elements to the `HT`.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%HT(ADD,key=x,data=17)
%HT(ADD,key=y,data=42)
%HT(ADD,key=z,data=303)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Add some duplicates for the key x. 
  See macrovariables created.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%HT(ADD,key=x,data=18) 
%HT(ADD,key=x,data=19)

%put _user_;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Check the number od data portions in macrohash
  for the key `x` and non existing key `t`.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put ##%HT(DP,key=x)##;
%put ##%HT(DP,key=t)##;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Check the number od data portions in macrohash
  for the key index 1 and 4.
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put ##%HT(CHECKIDX,key=1)##;
%put ##%HT(CHECKIDX,key=4)##;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Prints first data values for various keys.
  Key `t` does not exist in the macrohash.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put #%HT(FIND,key=x)#;
%put #%HT(FIND,key=y)#;
%put #%HT(FIND,key=z)#;
%put #%HT(FIND,key=t)#;

%put #%HT(FIND,key=x,data=2)#;
%put #%HT(FIND,key=x,data=3)#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Print first and subsequent data values 
  for a given KeyIDX. Index `4` does not exist.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put #%HT(KEYIDX,key=1)#;
%put #%HT(KEYIDX,key=2)#;
%put #%HT(KEYIDX,key=3)#;
%put #%HT(KEYIDX,key=4)#;

%put #%HT(KEYIDX,key=1,data=2)#;
%put #%HT(KEYIDX,key=1,data=3)#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Print the key values for a given KeyIDX.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%put #%HT(KEYVAL,key=1)#;
%put #%HT(KEYVAL,key=2)#;
%put #%HT(KEYVAL,key=3)#;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

  Clear and delete macro hash table `HT`.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%HT(CLEAR)
%mcHashTable(HT,DELETE)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Combine `CHECK` and `FIND` methods
  with macros `%array()` and `%do_over()`

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(H)
%H(ADD,key=x,data=17)
%H(ADD,key=x,data=18) 
%H(ADD,key=x,data=19)

%array(A[%H(CHECK,key=x)]);

%put %do_over(A, phrase=%nrstr(
  %H(FIND,key=x,data=&_i_)
), between = %str(,));

%mcHashTable(H,delete)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 3.** Populate macro hash table from a dataset.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(CLASS)
%let t = %sysfunc(datetime());
data _null_;
  set sashelp.class;
  call execute('%CLASS(ADD,key=' !! name !! ',data=' !! age !! ')');
  call execute('%CLASS(ADD,key=' !! name !! ',data=' !! weight !! ')');
  call execute('%CLASS(ADD,key=' !! name !! ',data=' !! height !! ')');
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put _user_;
%CLASS(CLEAR)


%mcHashTable(CARS)
%let t = %sysfunc(datetime());
data _null_;
  set sashelp.cars;
  call execute('%CARS(ADD,key=' !! catx("|",make,model)  !! ',data=' !! MPG_CITY !! ')');
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%* %put _user_;
%CARS(CLEAR)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 4.** Data portion may require quoting and un-quoting..

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(CODE)
%CODE(CLEAR)
%CODE(ADD,key=data, data=%str(data test; x = 42; run;))
%CODE(ADD,key=proc, data=%str(proc print; run;))
%CODE(ADD,key=macro,data=%nrstr(%put *****;))

%CODE(FIND,key=data)
%CODE(FIND,key=proc) 
%unquote(%CODE(FIND,key=macro))

%mcHashTable(CODE,DELETE)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 5.** Longer lists.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%let size = 1000;

%mcHashTable(AAA)
%mcHashTable(BBB)
%mcHashTable(CCC)
%mcHashTable(DDD)

%let t = %sysfunc(datetime());
data _null_;
  do i = 1 to &size.;
    call execute(cats('%AAA(ADD,key=A', i, ',data=', i, ')'));
  end;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=AAA_KEYSNUM;
%AAA(CLEAR)

%let t = %sysfunc(datetime());
data _null_;
  do i = 1 to &size.;
    call execute(cats('%BBB(ADD,key=B', i, ',data=', i, ')'));
    call execute(cats('%BBB(ADD,key=B', i, ',data=', i+1, ')'));
  end;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=BBB_KEYSNUM;
%BBB(CLEAR)

%let t = %sysfunc(datetime());
data _null_;
  t= datetime();
  do i = 1 to &size.;
    call execute(cats('%CCC(ADD,key=C', i, ',data=', i, ')'));
  end;
  t = datetime() - t;
  put t=;
  t= datetime();
  do i = 1 to &size.;
    call execute(cats('%CCC(ADD,key=C', i, ',data=', i+1, ')'));
  end;
  t = datetime() - t;
  put t=;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);

%let t = %sysfunc(datetime());
data test;
  do i = 1 to &size.;
    x = resolve(cats('%CCC(FIND,key=C', i, ',data=1)'));
    y = resolve(cats('%CCC(FIND,key=C', i, ',data=2)'));
    output;
  end;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=CCC_KEYSNUM;
%CCC(CLEAR)

%let t = %sysfunc(datetime());
data _null_;
  do i = 1 to &size.;
    call execute(cats('%DDD(ADD,key=D,data=', i, ')'));
  end;
run;
%put t = %sysevalf(%sysfunc(datetime()) - &t.);
%put &=DDD_KEYSNUM;
%put %DDD(CHECK,key=D);
%DDD(CLEAR)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 6.** Forbidden names.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable()
%mcHashTable(_)

%mcHashTable(ABCDEFGHIJK) %* bad;
%mcHashTable(ABCDEFGHIJ)  %* good;
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

**EXAMPLE 7.** Hashing algorithms.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas
%mcHashTable(H1,DCL,HASH=MD5)
%mcHashTable(H2,DECLARE,HASH=CRC32)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---
*//*** HELP END ***/

/**###################################################################**/
/*                                                                     */
/*  Copyright Bartosz Jablonski, since October 2020.                   */
/*                                                                     */
/*  Code is free and open source. If you want - you can use it.        */
/*  But it comes with absolutely no warranty whatsoever.               */
/*  If you cause any damage or something - it will be your own fault.  */
/*  You have been warned! You are using it on your own risk.           */
/*  However, if you decide to use it don't forget to mention author.   */
/*  Bartosz Jablonski (yabwon@gmail.com)                               */
/*                                                                     */
/**###################################################################**/
