/*** HELP START ***//*
 
## >>> `%sortMacroArray()` macro: <<< <a name="sortmacroarray-macro"></a> #######################  

The sortMacroArray() macro
allow to sort elements of a macro array. 

The **limitation** is that sorted values are limited to 32767 bytes of length.

See examples below for the details.

### SYNTAX: ###################################################################

The basic syntax is the following, the `<...>` means optional parameters:
~~~~~~~~~~~~~~~~~~~~~~~sas
%sortMacroArray(
    array
  <,valLength=>
  <,outSet=> 
  <,sortseq=>
)
~~~~~~~~~~~~~~~~~~~~~~~

**Arguments description**:

1. `array`             - *Required*, name of an array generated by the `%array()` macro.

* `valLength = 32767`  - *Optional*, default value is `32767`, 
                          maximum length of a variable storing macrovariable data.
                          (the reason of 32767 limitation) 

* `outSet    = _NULL_` - *Optional*, default value is `_NULL_`,
                          an optional output dataset name. 

* `sortseq  =`         - *Optional*, default value is `LINGUISTIC(NUMERIC_COLLATION = ON)`,
                          sorting options for use in an internal `Proc SORT`.

*//*** HELP END ***/

/* sortmacroarray.sas */

%macro sortMacroArray(
   array                                          /* array name */
  ,valLength = 32767                              /* length of a variable strorinh macrovariable data */
  ,outSet    = _NULL_                             /* optional output dataset */
  ,sortseq   = LINGUISTIC(NUMERIC_COLLATION = ON) /* sorting options */
)
/ secure
des='The %sortMacroArray() macro function allows to sort elements of an array.'
;
  %local _ds_ _TMP_options_ _tmp_syscc_;
  %let _TMP_options_ = %sysfunc(getoption(notes)) %sysfunc(getoption(source));
  %let _ds_ = work._%sysfunc(datetime(), hex16.)_;

  options nonotes nosource;

  %let _tmp_syscc_=&syscc.;
  %let syscc=0;

  data &_ds_.;
   set sashelp.vmacro;
   where scope = 'GLOBAL' 
     and name like "%upcase(&array.)" !! '%'
     and input(substr(name, %length(&array.)+1), ? best32.) between &&&array.LBOUND and &&&array.HBOUND
   ;
   drop scope;
  run; 
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: data collection.;

  proc sort data=&_ds_.;
    by name offset;
  run;
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: presorting.;

  data &_ds_.;
    set &_ds_.;
    by name;

    length val $ &valLength.;
    retain val "";
    if first.name then val = "";
    substr(val, offset+1) = value;
    if last.name then output;
    drop offset value name;
  run;
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: agregation.;

  proc sort data=&_ds_. SORTSEQ=&SORTSEQ.;
    by val;
  run;
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: sorting.;

  data &outSet.;
    retain _I_ &&&array.LBOUND;
    set &_ds_.;
    call symputx(cats("&array.", _I_), val, "G");
    output;
    _I_+1;
  run;
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: generating macrovariables.;

  proc delete data = &_ds_.;
  run;
  %if &syscc. %then %put ERROR:[&sysmacroname.] Error in sorting at: deleting.;

  %let syscc=&_tmp_syscc_.;
  options &_TMP_options_.;
%mend sortMacroArray;




/*** HELP START ***//*

### EXAMPLES AND USECASES: ####################################################


**EXAMPLE 1.** Basic use-case. 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas

options mprint;
ods html;
ods listing close;


%array(hij [4:9] $ 512 ("C33" "B22" "A11" "A01" "A02" "X42"), macarray=Y)

%put NOTE: %do_over(hij);

%sortMacroArray(hij, valLength=3, outSet = A_NULL_(compress=char))

%put NOTE: %do_over(hij);

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~


**EXAMPLE 2.** Basic use-case. 

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~sas

options mprint;
ods html;
ods listing close;


%array(ds = sashelp.class, vars = name|NNN height|h, macarray=Y)
%array(ds = sashelp.cars, vars = model|, macarray=Y)

%put NOTE: %do_over(NNN);
%put NOTE: %do_over(H);
%put NOTE: %do_over(model);

%sortMacroArray(NNN, valLength=30, outSet = A_NULL_(compress=char))
%sortMacroArray(H, valLength=32)
%sortMacroArray(model, valLength=120)

%put NOTE: %do_over(NNN);
%put NOTE: %do_over(H);
%put NOTE: %do_over(model);

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

---

*//*** HELP END ***/
